use serde_derive::{Deserialize, Serialize};
use crate::{interfaces,errors};
use nanoid::nanoid;
use crypto::sha1::Sha1;
use crypto::digest::Digest;
pub mod db_recovery;
use chrono::prelude::*;

#[macro_export]
macro_rules! blobid {
    ($n:expr) => { format!("BLOB{}",$n) }
}

#[macro_export]
macro_rules! tmpfmt {
    () => { "./tmp/{}" }
}

#[macro_export]
macro_rules! tmpzipfmt {
    ($n:expr) => { format!("./tmp/{}.zip",$n) }
}

#[macro_export]
macro_rules! tmppdffmt {
    ($n:expr) => { format!("./tmp/{}.pdf",$n) }
}

#[macro_export]
macro_rules! tmptxtfmt {
    ($n:expr) => { format!("./tmp/{}.txt",$n) }
}

#[macro_export]
macro_rules! tmpdownloadpdffmt {
    ($n:expr) => { format!("download/{}",$n) }
}


#[macro_export]
macro_rules! pdffmt{
    () => { "{}.pdf" }
}


#[macro_export]
macro_rules! zipfmt{
    () => { "{}.zip" }
}

pub fn generate_id() -> String{

    let mut hasher = Sha1::new();
    
    hasher.input_str(&nanoid!());

    hasher.result_str()

}

pub fn now_microseconds()-> i64{
    Utc::now().timestamp() * 1_000_000
}

pub fn convert_unix_timestamp_microseconds(i: i64) -> i64 {
    i * 1_000_000
}

pub fn convert_microseconds_unix_timestamp(i: i64) -> i64 {
    i / 1_000_000
}

pub fn date_time_from_utc_timestamp(i: i64) -> DateTime<FixedOffset>{

    let seconds = (i / 1_000_000) as i64;
    let microseconds = (i % 1_000_000) as u32;
    
    if let Some(ndt) = NaiveDateTime::from_timestamp_opt(seconds,microseconds){
	DateTime::from_utc(ndt, FixedOffset::west(5*3600))
    } else {
	log::info!("falla realizando conversion de timestamp a objeto DateTime. Valor fuente: {}",i);
	Utc::now().into()
    }
}

pub fn date_time_to_string(i: &DateTime<FixedOffset>) -> String{
    i.format("%F %R").to_string()
}

pub fn utc_timestamp_to_string(i: i64) -> String{
    date_time_from_utc_timestamp(i).to_rfc3339()
}


pub fn redondear_str_float(i: &str,decimales: i32) -> String {
    let potencia = 10f64.powi(decimales);
    let r :f64 = i.parse().unwrap_or(0.0f64).mul_add(potencia,0.0f64);
    (r.trunc() / 100.0).to_string()
}

pub fn generar_clave_dinamica() -> String{
    
    nanoid!(5, &['1','2','3','4','5','6','7','8','9','0'] )

}

#[derive(Serialize,Deserialize,Clone)]
pub struct ArchivoConfiguracion{
    pub cuenta_correo_electronico: String,
    pub clave_correo_electronico: String,
    pub nombre_host_correo: String,
    pub puerto_host_correo: u16,
    pub puerto_http: u16,
    pub tls_cert_name: String,
    pub tls_cert_account: String,
    pub tls_mode: bool,
    pub max_intentos_inicio_sesion: usize,
    pub api_key: String,
    pub url_proxy: String,
    pub firebase: ConfiguracionServidorFirebase,
    pub procesos_fondo: ConfiguracionProcesosFondo,
    pub servidor_backend_sicap: ConfiguracionBackendSICAP,
    pub programcion_limpieza_temporal: String,
}

#[derive(Serialize,Deserialize,Clone)]
pub struct ConfiguracionBackendSICAP{
    pub client_secret: String,
    pub client_id: String,
    pub token: String,
}

#[derive(Serialize,Deserialize,Clone)]
pub struct ConfiguracionServidorFirebase{
    pub api_key: String,
    pub url: String,
}

#[derive(Serialize,Deserialize,Clone)]
pub struct ConfiguracionProcesosFondo{
    pub cron_espera_consulta_sicap_notificaciones_push: String,
    pub tiempo_espera_ticks_motor_milisegundos: u64,
}


impl ArchivoConfiguracion{

    pub fn new() -> Self{
	Self {
	    cuenta_correo_electronico: "cuenta-envio@empresa.com".to_string(),
	    clave_correo_electronico: "abcdefghi".to_string(),
	    nombre_host_correo: "servidor.smtp.com".to_string(),
	    puerto_host_correo: 587,
	    puerto_http: 60000,
	    tls_cert_name: "portal-proveedores.empresa.com".to_string(),
	    tls_cert_account: "portal@email.xyz".to_string(),
	    tls_mode: false,
	    max_intentos_inicio_sesion: 4,
            api_key:"123456".to_string(),
            url_proxy: "http://x.y.z:9090".to_string(),
	    firebase: ConfiguracionServidorFirebase{
		api_key: "123456".to_string(),
		url: "https://".to_string(),
	    },
	    procesos_fondo: ConfiguracionProcesosFondo{
		cron_espera_consulta_sicap_notificaciones_push: "0 15 6,8,10 * Mar,Jun Fri 2017".to_string(),
		tiempo_espera_ticks_motor_milisegundos: 2,
	    },
            programcion_limpieza_temporal: "* * 1/6 * * * *".to_string(),
 	    servidor_backend_sicap:ConfiguracionBackendSICAP{
		client_secret: "secreto".to_string(),
		client_id: "id_de_cliente".to_string(),
		token: Default::default(),
	    }
	}
    }
    
    pub fn crear_config_ejemplo() -> Result<(), std::io::Error>{

	let configuracion = ArchivoConfiguracion {
	    cuenta_correo_electronico: "cuenta-envio@empresa.com".to_string(),
	    clave_correo_electronico: "abcdefghi".to_string(),
	    nombre_host_correo: "servidor.smtp.com".to_string(),
	    puerto_host_correo: 587,
	    puerto_http: 60000,
	    tls_cert_name: "portal-proveedores.empresa.com".to_string(),
	    tls_cert_account: "portal@email.xyz".to_string(),
	    tls_mode: false,
	    max_intentos_inicio_sesion: 4,
            api_key:"123456".to_string(),
            url_proxy: "http://x.y.z:9090".to_string(),
	    firebase: ConfiguracionServidorFirebase{
		api_key: "123456".to_string(),
		url: "https://".to_string(),
	    },
	    procesos_fondo: ConfiguracionProcesosFondo{
		cron_espera_consulta_sicap_notificaciones_push: "0 15 6,8,10 * Mar,Jun Fri 2017".to_string(),
		tiempo_espera_ticks_motor_milisegundos: 2,
	    },
            programcion_limpieza_temporal: "* * 1/6 * * * *".to_string(),
	    servidor_backend_sicap:ConfiguracionBackendSICAP{
		client_secret: "secreto".to_string(),
		client_id: "id_de_cliente".to_string(),
		token: Default::default(),
	    }

	};
	
	std::fs::write( "configuracion.toml", toml::to_vec(&configuracion).unwrap() )
    }

    pub fn escribir_configuracion(&self) -> Result<(), errors::ErroresConfiguracion>{

	let toml_value = toml::Value::try_from(self).map_err(|why| errors::ErroresConfiguracion::FallaInterpretacionTOML(why.to_string()) )?;
	let bytes = toml::ser::to_vec(&toml_value).map_err(|why| errors::ErroresConfiguracion::FallaInterpretacionTOML(why.to_string()) )?;
	
	std::fs::write(
	    "configuracion.toml",
	    bytes
	).map_err(|x| errors::ErroresConfiguracion::FallaLecturaArchivo( x.to_string() )  )

    }

    pub fn leer_configuracion() -> Result<ArchivoConfiguracion, errors::ErroresConfiguracion>{
	let out : ArchivoConfiguracion = toml::from_str(
	    
	    std::str::from_utf8(
		&std::fs::read( "configuracion.toml" ).map_err(|x| errors::ErroresConfiguracion::FallaLecturaArchivo(x.to_string())  )?
	    ).map_err(|_x| errors::ErroresConfiguracion::CodificacionUTF8Incorrecta )?
		
	).map_err(|x| errors::ErroresConfiguracion::FallaInterpretacionTOML( x.to_string() ) )?;

	Ok(out)
    }


}


pub fn enviar_correo(destino: &str, asunto: &str, cuerpo:&str) -> Result<(), Box<dyn std::error::Error>>{
    
    use lettre::smtp::{
	authentication::Credentials,
	SmtpTransport
    };
    use lettre_email::{Email, EmailBuilder};
    use lettre::{
        ClientSecurity, ClientTlsParameters, SmtpClient,
        Transport,
    };
    use native_tls::{Protocol, TlsConnector};

    let configuracion = ArchivoConfiguracion::leer_configuracion()?;

    let datos_servidor_correo = ( &configuracion.nombre_host_correo.clone()[..], configuracion.puerto_host_correo );

    dbg!(&cuerpo);
	std::fs::write( "email_carmelita.html", cuerpo );
    
    let status = std::process::Command::new("php")
    .args( &[ "envio_correo.php", &destino, &asunto, "email_carmelita.html" ])
    .status()?;

    
    /*
    let email: Email = EmailBuilder::new()
        .to(destino)
        .from(configuracion.cuenta_correo_electronico.clone())
        .subject(asunto)
        .html(cuerpo)
        .build()
	.map_err(|e| Box::new(errors::PasswordRefreshFail::ErrorConstruyendoCuerpoCorreo{ motivo:e.to_string() }))?;

    let tls_builder = TlsConnector::builder()
        .min_protocol_version(Some(Protocol::Tlsv10))
        .build()
	.map_err(|e| Box::new(errors::PasswordRefreshFail::ErrorProtocoloTLS{ motivo:e.to_string() }))?;

    let tls_parameters = ClientTlsParameters::new( configuracion.nombre_host_correo, tls_builder );

    let mut mailer: SmtpTransport = SmtpClient::new(datos_servidor_correo, ClientSecurity::Required(tls_parameters))
	.map_err(|e| Box::new(errors::PasswordRefreshFail::ErrorConexionServidorSMTP{ motivo:e.to_string() }))?
        .credentials(Credentials::new(
            configuracion.cuenta_correo_electronico,
            configuracion.clave_correo_electronico,
        ))
        .transport();

    mailer.send(email.into())
	.map_err(|e| Box::new(errors::PasswordRefreshFail::ErrorEnviandoCorreo{ motivo:e.to_string() }))?;

    mailer.close();*/

    Ok(())
}


/*pub fn revisar_comando_existe_os(comando:&str) -> Result<bool, Box<dyn std::error::Error>>{
    let output = std::process::Command::new("bash")
	.arg("-c")
	.arg(format!(r#""command -v {}""#, comando))
	.arg(comando)
	.output()
	.map_err(|why| Box::new(errors::Converciones::HerramientaNoEncontrada{herramienta:"sh".to_string(), motivo_interno: why.to_string() } ) )?;

    let success = output.status.success();
    
    Ok(success)
}*/

pub fn convertir_pdf_en_archivo_texto(ruta_pdf: &std::path::Path) -> Result<std::path::PathBuf, Box<dyn std::error::Error>>{

    //Revisar que el archivo pdf exista
    if ruta_pdf.exists() == false {
	return Err(Box::new(errors::Converciones::ArchivoFuenteNoEncontrado{nombre_archivo: ruta_pdf.to_string_lossy().to_string()}));
    }
    
    //Revisar que gs este presente en el sistema en el ambiente de trabajo
    //
    // Para instalar GhostScript en instalaciones debian:
    //
    // apt install ghostscript
    //
    
    //generar un nombre de arhivo de buffer temporal
    let name = tmppdffmt!(nanoid!());
    let path = std::path::Path::new(&name);

    //ejecutar la rutina de gs y extraer el resultado en el archivo temporal
    //gs -sDEVICE=txtwrite -o output.txt ~/Downloads/LIQUIDACIONES_CAÑAMATA_NOVIEMBRE.pdf
    let output = std::process::Command::new("gs")
	.arg("-sDEVICE=txtwrite")
	.arg("-o")
	.arg(path)
	.arg(ruta_pdf)
	.output()
	.map_err(|why| Box::new(errors::Converciones::GhostScriptNoPresente{ motivo_interno:  why.to_string() } ) )?;


    //devolver la ruta del archivo
    if output.status.success() == true{
	Ok(path.to_path_buf())
    } else {
	let stdout = std::str::from_utf8(&output.stdout[..]).unwrap_or("(sin datos en stdout)");
	let stderr = std::str::from_utf8(&output.stderr[..]).unwrap_or("(sin datos en stderr)");
	let consolidado_error = format!("\n\nstdout:{}\n\nstderr:{}", stdout, stderr);
	Err(Box::new(errors::Converciones::FalloEjecucionGhostScript{pdf: ruta_pdf.to_string_lossy().to_string(), archivo_texto: path.to_string_lossy().to_string(), motivo_interno: consolidado_error}))
    }
}

pub fn convertir_pdf_en_string(ruta_pdf: &std::path::Path) -> Result<String, Box<dyn std::error::Error>>{

    let pathbuf = convertir_pdf_en_archivo_texto(ruta_pdf)?;
    
    let output = std::fs::read_to_string(pathbuf.as_path())
	.map_err(|e| errors::Converciones::ArchivoFuenteNoEncontrado{nombre_archivo: pathbuf.to_string_lossy().to_string()})?;

    std::fs::remove_file(pathbuf.as_path()).unwrap_or(());
	
    Ok(output)
	
}


pub fn separar_pdf_en_paginas(ruta_pdf: &std::path::Path) -> Result<Vec<std::path::PathBuf>, Box<dyn std::error::Error>>{

    //Revisar que el archivo pdf exista
    if ruta_pdf.exists() == false {
	return Err(Box::new(errors::Converciones::ArchivoFuenteNoEncontrado{nombre_archivo: ruta_pdf.to_string_lossy().to_string()}));
    }

    
    //revisar que pdfseparate este presente en el sistema en el ambiente de trabajo
    //
    // Para instalar pdfseparate en instalaciones debian usar:
    //
    // apt install poppler-utils
    //
    let plantilla_nombre_pagina_sencilla : String = tmppdffmt!(generate_id() +"_%d");
    
    //ejecutar la rutina de pdfseparate y extraer el resultado muchos archivos temporales
    //pdfseparate pdf-sample.pdf sample-%d.pdf
    let output = std::process::Command::new("pdfseparate")
	.arg(ruta_pdf)
	.arg(&plantilla_nombre_pagina_sencilla)
	.output()
	.map_err(|why| Box::new(errors::Converciones::PdfSeparateNoPresente{motivo_interno: why.to_string() } ) )?;

    
    //verificar la correcta terminacion del proceso
    if output.status.success() == true{

	//Obtener el listado de archivos separados generados
	let patron_busqueda : String = plantilla_nombre_pagina_sencilla.replace("%d","*");
	let iterador_archivos = glob::glob(&patron_busqueda)
	    .map_err(|e| errors::Converciones::FalloBusquedaArchivosSeparados{pdf: ruta_pdf.to_string_lossy().to_string(),  patron_busqueda: patron_busqueda, motivo_interno: e.to_string() } )?;

	let rutas : Vec<std::path::PathBuf> = iterador_archivos.filter_map(|nodo|{
	    if let Ok(ruta) = nodo{
		Some(ruta)
	    }else{
		None
	    }
	}).collect();

	
	Ok(rutas)
	    
    } else {
	
	let stdout = std::str::from_utf8(&output.stdout[..]).unwrap_or("(sin datos en stdout)");
	let stderr = std::str::from_utf8(&output.stderr[..]).unwrap_or("(sin datos en stderr)");
	let consolidado_error = format!("\n\nstdout:{}\n\nstderr:{}", stdout, stderr);
	return Err(Box::new(errors::Converciones::FalloEjecucionPdfSeparate{pdf: ruta_pdf.to_string_lossy().to_string(), motivo_interno: consolidado_error}));
	
    }

}


use std::error::Error;

pub async fn ask_reconnect_proxy() -> Result<(), Box<dyn std::error::Error>>{

    let config = crate::utils::ArchivoConfiguracion::leer_configuracion()?;
    
    #[derive(Serialize,Deserialize,Debug)]
    pub struct ReconectarDB{
        API_KEY:String,
    }

    let peticion = ReconectarDB{
        API_KEY: config.api_key,
    };

    macro_rules! url_reconectar{ () => { "{url_proxy}/reconectar_db" }  };

    
    let response = match reqwest::Client::new()
	.post(&format!(url_reconectar!(), url_proxy = config.url_proxy))
	.json(&dbg!(peticion))
	.send()
	.await{
	    Ok(r) => Ok(r),
	    Err(e) => Err(Box::new( errors::FallaProxy::ErrorCreandoSolicitud{motivo_interno: format!("Intentado reconecta a la base de datos ocurre falla: {}",e)} )),
	}?;

    if response.status() != 200 {

	let status = response.status();
	let body = match response.text()
	    .await {
	    Ok(r) => Ok(r),
	    Err(e) => Err(Box::new(errors::FallaProxy::ErrorLeyendoResultado{peticion:format!("{}: {}", url_reconectar!().to_string(), "lectura del cuerpo"), motivo_interno: e.to_string()})),
	}?;
	
	return Err(Box::new(errors::FallaProxy::ErrorRemoto{peticion:format!("{}: {} {}", url_reconectar!().to_string(), "Codigo de error", status), motivo_interno: body}));
    }

    Ok(())

}

#[derive(Serialize, Deserialize, PartialEq)]
pub enum TipoError{
    ErrorBaseDatos,
    ErrorDeserializacionCuerpo,
    OtroError,
}


#[derive(Serialize,Deserialize)]
pub struct RespuestaError{
    pub tipo_error: TipoError,
    pub mensaje: String 
}


#[macro_export]
macro_rules! request_proxy {
    ($endpoint: literal, $url_proxy: expr, $peticion: expr, $tipo_respuesta: ty,$timeout:expr) => {
	{   
        let timeout_duration = std::time::Duration::new($timeout, 0);
	    let response = reqwest::Client::new()
		.post(&format!($endpoint,url_proxy = $url_proxy))
        .timeout(timeout_duration)
		.json(&dbg!($peticion))
		.send()
		.await.map_err(|e| interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorCreandoSolicitud{motivo_interno: e.to_string()})) )?;
	    
        dbg!(&response);
	    if response.status() == 200 {
		response
		    .json::<$tipo_respuesta>()
		    .await.map_err(|e|{			
			interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorLeyendoResultado{peticion:format!("{}: {}", $endpoint.to_string(), "falla deserializando resultado proxy"), motivo_interno: e.to_string()}))
		    } )?
		    
	    } else if response.status() == 400 {

		let e_response = response
		    .json::<crate::utils::RespuestaError>()
		    .await.map_err(|e| interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorLeyendoResultado{peticion:format!("{}: {}", $endpoint.to_string(), "leyendo respuesta de error"), motivo_interno: e.to_string()})))?;

		if e_response.tipo_error == crate::utils::TipoError::ErrorBaseDatos{

		    crate::utils::ask_reconnect_proxy()
			.await.map_err(|e| interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorLeyendoResultado{peticion:format!("{}: {}", $endpoint.to_string(), "leyendo respuesta de error al intentar reconexion base de datos"), motivo_interno: e.to_string()})))?;
		    
		    return interfaces::error_response(Box::new(errors::FallaProxy::ErrorConexionBaseDatos{motivo_interno: e_response.mensaje }));
		    
		}else{
		    
		    return interfaces::error_response(Box::new(errors::FallaProxy::ErrorRemoto{peticion:format!("{}: {}", $endpoint.to_string(), "mensaje de error remoto"), motivo_interno: e_response.mensaje}));
		    
		}

	    }else{

		let status = response.status();
		let body = response.text()
		    .await.map_err(|e| interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorLeyendoResultado{peticion:format!("{}: {}", $endpoint.to_string(), "deserializando resultado fallido"), motivo_interno: e.to_string()})) )?;
		
		return interfaces::error_response(Box::new(errors::FallaProxy::ErrorRemoto{peticion:format!("{}: {} {}", $endpoint.to_string(), "Codigo de error", status), motivo_interno: body}));
	    }
	}
    };
}

#[macro_export]
macro_rules! refresh_token {
    ($timeout: expr) => {
	{   
            let timeout_duration = std::time::Duration::new($timeout, 0);
	    let mut archivo_configuracion = ArchivoConfiguracion::leer_configuracion().map_err(|why| crate::interfaces::error_response_map(Box::new(why)) )?;

	    let ArchivoConfiguracion{ servidor_backend_sicap: ConfiguracionBackendSICAP{
		client_secret,
		client_id,
		token: _token,
	    }, .. } = &archivo_configuracion;
	    
	    let peticion = serde_json::json!(
		{
		    "client_secret": client_secret,
		    "grant_type": "client_credentials",
		    "client_id": client_id,
		    "scope": "informes"
		}
	    );
	    
	    let response = reqwest::Client::new()
		.post(&format!("{url_proxy}/Auth",url_proxy = archivo_configuracion.url_proxy))
		.timeout(timeout_duration)
		.json(&dbg!(peticion))
		.send()
		.await.map_err(|e| interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorCreandoSolicitud{motivo_interno: e.to_string()})) )?;
	    
            dbg!(&response);
	    
	    if response.status() == 200 {
		
		let response_data = response
		    .json().await.map_err(|e|{			
			interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorLeyendoResultado{peticion:format!("{}: {}", "", "falla deserializando resultado proxy"), motivo_interno: e.to_string()}))
		    } )?;

		if let serde_json::Value::Object(object) = response_data{
		    if let Some(serde_json::Value::String(token)) = object.get("access_token"){
			archivo_configuracion.servidor_backend_sicap.token = token.to_string();
		    }
		}

		archivo_configuracion.escribir_configuracion().map_err(|why| crate::interfaces::error_response_map(Box::new(why)) )?;
		
	    } else if response.status() == 400 {
		
		let e_response = response
		    .json::<crate::utils::RespuestaError>()
		    .await.map_err(|e| interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorLeyendoResultado{peticion:format!("{}: {}", "", "leyendo respuesta de error"), motivo_interno: e.to_string()})))?;

		if e_response.tipo_error == crate::utils::TipoError::ErrorBaseDatos{
		    
		    crate::utils::ask_reconnect_proxy()
			.await.map_err(|e| interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorLeyendoResultado{peticion:format!("{}: {}", "", "leyendo respuesta de error al intentar reconexion base de datos"), motivo_interno: e.to_string()})))?;
		    
		    return interfaces::error_response(Box::new(errors::FallaProxy::ErrorConexionBaseDatos{motivo_interno: e_response.mensaje }));
		    
		}else{
		    
		    return interfaces::error_response(Box::new(errors::FallaProxy::ErrorRemoto{peticion:format!("{}: {}", "", "mensaje de error remoto"), motivo_interno: e_response.mensaje}));
		    
		}

	    }else{

		let status = response.status();
		let body = response.text()
		    .await.map_err(|e| interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorLeyendoResultado{peticion:format!("{}: {}", "", "deserializando resultado fallido"), motivo_interno: e.to_string()})) )?;
		
		return interfaces::error_response(Box::new(errors::FallaProxy::ErrorRemoto{peticion:format!("{}: {} {}", "", "Codigo de error", status), motivo_interno: body}));
	    }
	}
    };
}

#[macro_export]
macro_rules! request_proxy_authenticated {
    ($url_proxy: expr,$tipo_respuesta: ty, $timeout:expr) => {
	{   
            let timeout_duration = std::time::Duration::new($timeout, 0);
	    let mut headers = reqwest::header::HeaderMap::new();

	    let mut archivo_configuracion = ArchivoConfiguracion::leer_configuracion().map_err(|why| crate::interfaces::error_response_map(Box::new(why)) )?;

	    let ArchivoConfiguracion{ servidor_backend_sicap: ConfiguracionBackendSICAP{
		client_secret: _client_secret,
		client_id: _client_id,
		token
	    }, .. } = &archivo_configuracion;
	    
	    headers.insert("Authorization",reqwest::header::HeaderValue::from_str(&format!("Bearer {token}")).unwrap());
	    
	    let response = reqwest::Client::builder()
		.default_headers(headers)
		.build().map_err(|why| crate::interfaces::error_response_map(Box::new(why)) )?
		.get($url_proxy)
                .timeout(timeout_duration)
		.send()
		.await.map_err(|e| interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorCreandoSolicitud{motivo_interno: e.to_string()})) )?;
	    
            dbg!(&response);
	    if response.status() == 200 {
		
		response
			.json::<$tipo_respuesta>()
		    .await.map_err(|e|{			
			interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorLeyendoResultado{peticion:format!("{}: {}", "", "falla deserializando resultado proxy"), motivo_interno: e.to_string()}))
		    } )?
		    
	    } else if response.status() == 400 {

		response
		    .json()
		    .await.map_err(|e|{			
			interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorLeyendoResultado{peticion:format!("{}: {}", "", "falla deserializando resultado proxy"), motivo_interno: e.to_string()}))
		    } )?
		
	    }else{

		let status = response.status();
		let body = response.text()
		    .await.map_err(|e| interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorLeyendoResultado{peticion:format!("{}: {}", "", "deserializando resultado fallido"), motivo_interno: e.to_string()})) )?;
		
		return interfaces::error_response(Box::new(errors::FallaProxy::ErrorRemoto{peticion:format!("{}: {} {}", "", "Codigo de error", status), motivo_interno: body}));
	    }
	}
    };
}


#[macro_export]
macro_rules! request_proxy_authenticated_download {
    ($url_proxy: expr, $timeout:expr, $ext: expr ) => {
	{   
            let timeout_duration = std::time::Duration::new($timeout, 0);
	    let mut headers = reqwest::header::HeaderMap::new();

	    let mut archivo_configuracion = ArchivoConfiguracion::leer_configuracion().map_err(|why| crate::interfaces::error_response_map(Box::new(why)) )?;

	    let ArchivoConfiguracion{ servidor_backend_sicap: ConfiguracionBackendSICAP{
		client_secret: _client_secret,
		client_id: _client_id,
		token
	    }, .. } = &archivo_configuracion;
	    
	    headers.insert("Authorization",reqwest::header::HeaderValue::from_str(&format!("Bearer {token}")).unwrap());
	    
	    let response = reqwest::Client::builder()
		.default_headers(headers)
		.build().map_err(|why| crate::interfaces::error_response_map(Box::new(why)) )?
		.get($url_proxy)
                .timeout(timeout_duration)
		.send()
		.await.map_err(|e| interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorCreandoSolicitud{motivo_interno: e.to_string()})) )?;
	    
            dbg!(&response);
	    if response.status() == 200 {

		let nombre_archivo = crate::utils::generate_id();	
		let bytes_archivo : Vec<u8> = 
		    response
		    .bytes()
		    .await.map_err(|e|{			
			interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorLeyendoResultado{peticion:format!("{}: {}", "", "falla deserializando resultado proxy"), motivo_interno: e.to_string()}))
		    } )?
		    .into_iter().collect();
		
		let mut ruta = std::path::PathBuf::from( "./tmp");
		ruta.push(&nombre_archivo);
		ruta.set_extension($ext);
		let _ = std::fs::write( ruta , &bytes_archivo   );

		Ok::<_, Box<dyn std::error::Error>>(serde_json::json!({"ruta_descargar": format!("/download/{}.pdf",nombre_archivo)}))


	    } else{

		let status = response.status();
		let body = response.text()
		    .await.map_err(|e| interfaces::error_response_map(Box::new(errors::FallaProxy::ErrorLeyendoResultado{peticion:format!("{}: {}", "", "deserializando resultado fallido"), motivo_interno: e.to_string()})) )?;
		
		return interfaces::error_response(Box::new(errors::FallaProxy::ErrorRemoto{peticion:format!("{}: {} {}", "", "Codigo de error", status), motivo_interno: body}));
	    }
	}
    };
}

fn convertir_char_latinos_en_mayus(val: &str) -> String{
    val.chars().map(|x| {
        match x {
	    'á' => 'Á',
	    'é' => 'É',
	    'í' => 'Í',
	    'ó' => 'Ó',
	    'ú' => 'Ú',
	    'ñ' => 'Ñ',
	    'ü' => 'Ü',
	    _ => x,
	    }
    }).collect::<String>()
}


pub fn icontiene(haystack: &str, needle: &str) -> bool{
    
    convertir_char_latinos_en_mayus(haystack)
	.to_uppercase()
	.as_str()
	.contains(
	    convertir_char_latinos_en_mayus(needle)
		.to_uppercase()
		.as_str()
	)
}

pub fn ientre_rango<T>(valor:T, limite_inferior:T, limite_superior:T) -> bool
    where T : PartialEq + PartialOrd
{
    if valor >= limite_inferior && valor <= limite_superior {
	true
    }else{
	false
    }
}

pub fn entre_rango<T>(valor:T, limite_inferior:T, limite_superior:T) -> bool
where
    T : PartialEq + PartialOrd
{
    if valor > limite_inferior && valor < limite_superior {
	true
    }else{
	false
    }
}

pub fn guardar_blob(db: &std::sync::Arc<sled::Db>, id:&str, blob: Vec<u8>) -> Result<(), Box<dyn std::error::Error>> {

    let bid = blobid!(id);
    
    let _ = db.insert(&bid[..], &blob[..]).map_err(|e| Box::new( errors::StorableError::WriteError(e.to_string() )))?;

    Ok(())
}

pub fn leer_blob(db: &std::sync::Arc<sled::Db>, id: &str) -> Result<sled::IVec, Box<dyn std::error::Error>> {

    let bid = blobid!(id);
    
    let blob = db.get(&bid[..])?.ok_or(Box::new(errors::StorableError::NotFound( id.to_string() )))?;

    Ok(blob)
}

pub fn borrar_blob(db: &std::sync::Arc<sled::Db>, id: &str) -> Result<(), Box<dyn std::error::Error>> {

    let bid = blobid!(id);
    
    let _ = db.remove(&bid[..])?.ok_or(Box::new(errors::StorableError::NotFound( id.to_string() )))?;

    Ok(())
}
