use serde::{Serialize};
use warp::http::Uri;
use warp::{Filter, Rejection, Reply};

macro_rules ! sesion {
    ($db_filter: tt) => {
	warp::header::value("Authorization")
	    .and( $db_filter.clone() )
	    .map(|authorization_header: HeaderValue, db: std::sync::Arc<sled::Db>| crate::handlers::cargar_sesion_desde_id( authorization_header.to_str().unwrap(), db.clone() ).expect("falta sesion") )
    }
}

pub fn tree(db: std::sync::Arc<sled::Db> , tls_cert_name: &str , tls_cert_account: &str) -> impl Filter<Extract = (impl Reply,), Error = std::convert::Infallible> + Clone {

    let configuracion = crate::utils::ArchivoConfiguracion::leer_configuracion().expect("Archivo de configuracion no pudo ser leido");
    let estados_proceso : std::sync::Arc<std::sync::Mutex<std::collections::HashMap<String,String>>> = std::sync::Arc::new(std::sync::Mutex::new(std::collections::HashMap::new()));
    
    let db_arc = db.clone();
    let db_filter = warp::any().map(move || db_arc.clone());
    let config_filter = warp::any().map(move|| configuracion.clone());
    let estados_proceso_filter = warp::any().map(move || estados_proceso.clone());

    let check_authenticated = warp::header::value("Authorization").and(db_filter.clone()).and_then( crate::handlers::check_authentication ).untuple_one();

    let tls_cert_name_arc = std::sync::Arc::new(tls_cert_name.to_string());
    let tls_cert_name_filter = warp::any().map(move || std::sync::Arc::clone(&tls_cert_name_arc));

    let tls_cert_account_arc = std::sync::Arc::new(tls_cert_account.to_string());
    let tls_cert_account_filter = warp::any().map(move || std::sync::Arc::clone(&tls_cert_account_arc));

    let log = warp::log("all");

    use warp::{http::header::HeaderValue};

    let usuarios_crear_usuario = warp::any()
	.and( warp::path("crear_usuario") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::crear_usuario );

    let usuarios_actualizar_usuario = warp::any()
	.and( warp::path("actualizar_usuario") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::actualizar_usuario );

    let usuarios_cargar_usuario = warp::any()
	.and( warp::path("cargar_usuario") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::cargar_usuario );


    let usuarios_listar_usuarios = warp::any()
	.and( warp::path("listar_usuarios") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::listar_usuarios );

    let usuarios_eliminar_usuario = warp::any()
	.and( warp::path("eliminar_usuario") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::eliminar_usuario );


    let usuarios_objetos_seguridad_menu = warp::any()
	.and( warp::path("objetos_seguridad_menu") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::objetos_seguridad_menu );

    let usuarios_objetos_seguridad = warp::any()
	.and( warp::path("objetos_seguridad") )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::objetos_seguridad );

    
    let listar_roles_asignados_usuario = warp::any()
	.and( warp::path("listar_roles_asignados_usuario") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::listar_roles_asignados_usuario );

    let listar_roles_noasignados_usuario = warp::any()
	.and( warp::path("listar_roles_noasignados_usuario") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::listar_roles_noasignados_usuario );

	let usuario_activo_or_eliminado = warp::any()
	.and( warp::path("usuario_activo_or_eliminado") )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::usuario_activo_or_eliminado );
	

    let asignar_rol_usuario = warp::any()
	.and( warp::path("asignar_rol_usuario") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::asignar_rol_usuario );

    let remover_rol_usuario = warp::any()
	.and( warp::path("remover_rol_usuario") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::remover_rol_usuario );

	let reporte_trazabilidad_acciones_usuarios = warp::any()
	.and( warp::path("reporte_trazabilidad_acciones_usuarios") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::reporte_trazabilidad_acciones_usuarios );

	let reporte_politica = warp::any()
	.and( warp::path("reporte_politica") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::reporte_politica );


    let validar_sesion =  warp::any()
	.and( warp::path("validar_sesion") )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::validar_sesion );

    let usuarios_1 = warp::post()
	.and( warp::path("usuarios") )
	.and(
	    usuarios_crear_usuario
		.or( usuarios_actualizar_usuario )
		.or( usuarios_cargar_usuario )
		.or( usuarios_listar_usuarios )
		.or( usuarios_objetos_seguridad_menu )

	);

    let usuarios_2 = warp::post()
	.and( warp::path("usuarios") )
	.and(
	    usuarios_eliminar_usuario
		.or( listar_roles_asignados_usuario )
		.or( listar_roles_noasignados_usuario )
		.or( validar_sesion )
		.or( asignar_rol_usuario )
	);


    let usuarios_3 = warp::post()
	.and( warp::path("usuarios") )
	.and( remover_rol_usuario
	      .or( usuarios_objetos_seguridad )
		  .or( reporte_trazabilidad_acciones_usuarios )
		  .or( reporte_politica )
		  .or( usuario_activo_or_eliminado )
	);

    let roles_crear_rol = warp::any()
	.and( warp::path("crear_rol") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::roles::crear_rol );

    let roles_actualizar_rol = warp::any()
	.and( warp::path("actualizar_rol") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::roles::actualizar_rol );

    let roles_eliminar_roles = warp::any()
	.and( warp::path("eliminar_roles") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::roles::eliminar_roles );

    let roles_cargar_rol = warp::any()
	.and( warp::path("cargar_rol") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::roles::cargar_rol );

    let roles_listar_roles = warp::any()
	.and( warp::path("listar_roles") )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::roles::listar_roles );

    let roles_listar_objetos_asignados_rol = warp::any()
	.and( warp::path("listar_objetos_asignados_rol") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::roles::listar_objetos_asignados_rol );

    let roles_listar_objetos_noasignados_rol = warp::any()
	.and( warp::path("listar_objetos_noasignados_rol") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::roles::listar_objetos_noasignados_rol );


    let roles_asignar_objetos_rol = warp::any()
	.and( warp::path("asignar_objetos_rol") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::roles::asignar_objetos_rol );

    let roles_remover_objetos_rol = warp::any()
	.and( warp::path("remover_objetos_rol") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::roles::remover_objetos_rol );



    let roles_1 = warp::post()
	.and( warp::path("roles") )
	.and(
	    roles_crear_rol
		.or( roles_actualizar_rol )
		.or( roles_eliminar_roles )
		.or( roles_cargar_rol )
		.or( roles_listar_roles )
	);

    let roles_2 = warp::post()
	.and( warp::path("roles") )
	.and( roles_listar_objetos_asignados_rol
	      .or( roles_listar_objetos_noasignados_rol )
	      .or( roles_asignar_objetos_rol )
	      .or( roles_remover_objetos_rol )
	);

    let admin_documentos_crear_orden = warp::any()
	.and( warp::path("crear_orden") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and_then( crate::handlers::admin_documentos::crear_orden );

    let admin_documentos_cargar_fuente_en_orden = warp::any()
	.and( warp::path("cargar_fuente_en_orden") )
	.and( warp::multipart::form().max_length(5_000_000) )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::admin_documentos::cargar_fuente_en_orden );

    let admin_documentos_procesar_orden = warp::any()
	.and( warp::path("procesar_orden") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and( estados_proceso_filter.clone() )
	.and_then( crate::handlers::admin_documentos::procesar_orden );

    let admin_documentos_consulta_progreso_orden = warp::any()
	.and( warp::path("consulta_progreso_orden") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and( estados_proceso_filter.clone() )
	.and_then( crate::handlers::admin_documentos::consulta_progreso_orden );

    let admin_documentos_consultar_documentos_producidos = warp::any()
	.and( warp::path("consultar_documentos_producidos") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and_then( crate::handlers::admin_documentos::consultar_documentos_producidos );

    let admin_documentos_distribuir_orden = warp::any()
	.and( warp::path("distribuir_orden") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and( config_filter.clone() )
	.and_then( crate::handlers::admin_documentos::distribuir_orden );

    let admin_documentos_eliminar_orden = warp::any()
	.and( warp::path("eliminar_orden") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and_then( crate::handlers::admin_documentos::eliminar_orden );

    let admin_documentos_listar_ordenes = warp::any()
	.and( warp::path("listar_ordenes") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and_then( crate::handlers::admin_documentos::listar_ordenes );

    let admin_documentos_listar_documentos = warp::any()
	.and( warp::path("listar_documentos") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and_then( crate::handlers::admin_documentos::listar_documentos );

    let admin_documentos_obtener_documento = warp::any()
	.and( warp::path("obtener_documento") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and_then( crate::handlers::admin_documentos::obtener_documento );

    let admin_documentos_editar_orden = warp::any()
	.and( warp::path("editar_orden") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and_then( crate::handlers::admin_documentos::editar_orden );
    
	let admin_pqrs= warp::any()
	.and( warp::path("crear_pqr") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and_then( crate::handlers::pqrs_admin::crear_pqr );

	let admin_listado_pqrs= warp::any()
	.and( warp::path("listar_pqr") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and_then( crate::handlers::pqrs_admin::listar_pqr );

	let admin_eliminar_pqrs= warp::any()
	.and( warp::path("eliminar_pqr") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and_then( crate::handlers::pqrs_admin::eliminar_pqr );
	
	let admin_editar_pqrs= warp::any()
	.and( warp::path("editar_pqr") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and_then( crate::handlers::pqrs_admin::editar_pqr );

	let admin_listado_toda_pqrs= warp::any()
	.and( warp::path("listar_todas_pqr") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and_then( crate::handlers::pqrs_admin::listar_todas_pqr );

	let admin_cambiar_estado_pqrs= warp::any()
	.and( warp::path("editar_estado_pqr") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and_then( crate::handlers::pqrs_admin::editar_estado_pqr );

    let admin_documentos_1 = warp::post()
	.and( warp::path("admin_documentos") )
	.and(
	    admin_documentos_crear_orden
		.or( admin_documentos_cargar_fuente_en_orden )
		.or( admin_documentos_procesar_orden )
		.or( admin_documentos_consulta_progreso_orden )
		.or( admin_documentos_consultar_documentos_producidos )
	);

    let admin_documentos_2 = warp::post()
	.and( warp::path("admin_documentos") )
	.and( admin_documentos_distribuir_orden 
		.or( admin_documentos_eliminar_orden )
		.or( admin_documentos_listar_ordenes )
		.or( admin_documentos_listar_documentos )
		.or( admin_documentos_obtener_documento )
	);


    let admin_documentos_3 = warp::post()
	.and( warp::path("admin_documentos") )
	.and( admin_documentos_editar_orden
	);

	let pqrs_admin = warp::post()
	.and( warp::path("pqrs_admin") )
	.and( admin_pqrs
		.or( admin_listado_pqrs )
		.or( admin_eliminar_pqrs )
		.or( admin_editar_pqrs )
		.or( admin_listado_toda_pqrs )
		.or( admin_cambiar_estado_pqrs )
	);
    
    
    let entrada_cana_listar_haciendas = warp::any()
	.and( warp::path("listar_haciendas") )
        .and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
        .and( config_filter.clone() )
	.and_then( crate::handlers::entrada_cana::listar_haciendas );
    
    let entrada_cana_listar_entrada_cana_general = warp::any()
	.and( warp::path("listar_entrada_cana_general") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
        .and( config_filter.clone() )
	.and_then( crate::handlers::entrada_cana::listar_entrada_cana_general );
 
    let entrada_cana_listar_entrada_cana_detalle  = warp::any()
	.and( warp::path("listar_entrada_cana_detalle") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
        .and( config_filter.clone() )
	.and_then( crate::handlers::entrada_cana::listar_entrada_cana_detalle );
 
    let entrada_cana_total_general  = warp::any()
	.and( warp::path("total_general") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
        .and( config_filter.clone() )
	.and_then( crate::handlers::entrada_cana::total_general );
  
    
    let entrada_cana = warp::post()
	.and( warp::path("entrada_cana") )
	.and(
	    entrada_cana_listar_entrada_cana_general
		.or ( entrada_cana_listar_entrada_cana_detalle )
		.or ( entrada_cana_total_general )
		.or ( entrada_cana_listar_haciendas )
	);


    let informe_produccion_consultar_info_produc_suerte  = warp::any()
	.and( warp::path("consultar_info_produc_suerte") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
        .and( config_filter.clone() )
	.and_then( crate::handlers::informe_produccion::consultar_info_produc_suerte );

    let informe_produccion_totales_info_produc_suerte  = warp::any()
	.and( warp::path("totales_info_produc_suerte") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
        .and( config_filter.clone() )
	.and_then( crate::handlers::informe_produccion::totales_info_produc_suerte );

    let informe_produccion_generar_info_pdf  = warp::any()
	.and( warp::path("generar_info_pdf") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
	.and_then( crate::handlers::informe_produccion::generar_info_pdf );


    let informe_produccion = warp::post()
	.and( warp::path("informe_produccion") )
	.and(
	    informe_produccion_consultar_info_produc_suerte
		.or ( informe_produccion_totales_info_produc_suerte )
		.or ( informe_produccion_generar_info_pdf )
	);



    let liquidaciones_cana_listar_liquidacion = warp::any()
	.and( warp::path("listar_liquidacion") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
        .and( config_filter.clone() )
	.and_then( crate::handlers::liquidaciones_cana::listar_liquidacion );
 
    let liquidaciones_cana_listar_anticipos = warp::any()
	.and( warp::path("listar_anticipos") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
        .and( config_filter.clone() )
	.and_then( crate::handlers::liquidaciones_cana::listar_anticipos );
 
    let liquidaciones_cana_listar_ajuste_merc_exceden = warp::any()
	.and( warp::path("listar_ajuste_merc_exceden") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and(db_filter.clone() )
        .and( config_filter.clone() )
	.and_then( crate::handlers::liquidaciones_cana::listar_ajuste_merc_exceden );	      
	      
    let liquidaciones_cana = warp::post()
	.and( warp::path("liquidaciones_cana") )
	.and( liquidaciones_cana_listar_liquidacion
	      .or( liquidaciones_cana_listar_anticipos )
	      .or( liquidaciones_cana_listar_ajuste_merc_exceden )
	);


   
    let cronologico_listar_cronologico = warp::any()
	.and( warp::path("listar_cronologico") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
        .and( config_filter.clone() )
	.and_then( crate::handlers::cronologico::listar_cronologico );
    
    let cronologico = warp::post()
	.and( warp::path("cronologico") )
	.and(
	    cronologico_listar_cronologico
	);


    let asistencia_tecnica_recuperar_texto = warp::any()
	.and( warp::path("recuperar_texto") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
        .and( config_filter.clone() )
	.and_then( crate::handlers::asistencia_tecnica::recuperar_texto );

    let asistencia_tecnica_almacenar_texto = warp::any()
	.and( warp::path("almacenar_texto") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
        .and( config_filter.clone() )
	.and_then( crate::handlers::asistencia_tecnica::almacenar_texto );
    
    let asistencia_tecnica = warp::post()
	.and( warp::path("asistencia_tecnica") )
	.and(
	    asistencia_tecnica_recuperar_texto
		.or( asistencia_tecnica_almacenar_texto )
	);

    
    let formato_vinculacion_juridica_cargar_pdf = warp::any()
	.and( warp::path("cargar_pdf") )
	.and( warp::multipart::form().max_length(5_000_000) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::formato_vinculacion_juridica::cargar_pdf );

    let formato_vinculacion_juridica_obtener_pdf = warp::any()
	.and( warp::path("obtener_pdf") )
	.and( db_filter.clone() )
	.and_then( crate::handlers::formato_vinculacion_juridica::obtener_pdf );
    
    let formato_vinculacion_juridica = warp::post()
	.and( warp::path("formato_vinculacion_juridica") )
	.and(
	    formato_vinculacion_juridica_cargar_pdf
		.or( formato_vinculacion_juridica_obtener_pdf )
	);
    
    let formato_vinculacion_natural_cargar_pdf = warp::any()
	.and( warp::path("cargar_pdf") )
	.and( warp::multipart::form().max_length(5_000_000) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::formato_vinculacion_natural::cargar_pdf );

    let formato_vinculacion_natural_obtener_pdf = warp::any()
	.and( warp::path("obtener_pdf") )
	.and( db_filter.clone() )
	.and_then( crate::handlers::formato_vinculacion_natural::obtener_pdf );
    
    let formato_vinculacion_natural = warp::post()
	.and( warp::path("formato_vinculacion_natural") )
	.and(
	    formato_vinculacion_natural_cargar_pdf
		.or( formato_vinculacion_natural_obtener_pdf )
	);

    let formato_autorizacion_tratamiento_cargar_pdf = warp::any()
	.and( warp::path("cargar_pdf") )
	.and( warp::multipart::form().max_length(5_000_000) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::formato_autorizacion_tratamiento::cargar_pdf );

    let formato_autorizacion_tratamiento_obtener_pdf = warp::any()
	.and( warp::path("obtener_pdf") )
	.and( db_filter.clone() )
	.and_then( crate::handlers::formato_autorizacion_tratamiento::obtener_pdf );
    
    let formato_autorizacion_tratamiento = warp::post()
	.and( warp::path("formato_autorizacion_tratamiento") )
	.and(
	    formato_autorizacion_tratamiento_cargar_pdf
		.or( formato_autorizacion_tratamiento_obtener_pdf )
	);



    let autenticar = warp::post()
	.and( warp::path("autenticar") )
	.and( warp::body::json() )
	.and( db_filter.clone() )
	.and_then( crate::handlers::autenticar );

    let actualizar_token = warp::post()
	.and( warp::path("actualizar-token") )
	.and( warp::body::json() )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
        .and( config_filter.clone() )
	.and_then( crate::handlers::actualizar_token );

	let establecer_aceptacion_politica_admin_datos = warp::post()
	.and( warp::path("establecer_aceptacion_politica_admin_datos") )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
        .and( config_filter.clone() )
	.and_then( crate::handlers::establecer_aceptacion_politica_admin_datos );

	let limpiar_aceptacion_politica_admin_datos = warp::post()
	.and( warp::path("limpiar_aceptacion_politica_admin_datos") )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
        .and( config_filter.clone() )
	.and_then( crate::handlers::limpiar_aceptacion_politica_admin_datos );

	// let roles_actualizar_rol = warp::any()
	// .and( warp::path("consulta") )
	// .and( warp::body::json() )
	// .and( sesion!(db_filter) )
	// .and( db_filter.clone() )
	// .and_then( crate::handlers::tra::consulta );
    
    let solicitar_recuperar_clave = warp::post()
	.and( warp::path("solicitar_recuperar_clave") )
	.and( warp::body::json() )
	.and( db_filter.clone() )
	.and_then( crate::handlers::solicitar_recuperar_clave );

    let recuperar_clave = warp::post()
	.and( warp::path("recuperar_clave") )
	.and( warp::body::json() )
	.and( db_filter.clone() )
	.and_then( crate::handlers::recuperar_clave );

    let cerrar_sesion = warp::post()
	.and( warp::path("cerrar_sesion") )
	.and( warp::header::value("Authorization") )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::cerrar_sesion );

    let usuario_actual = warp::post()
	.and( warp::path("usuario_actual") )
	.and( sesion!(db_filter) )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::usuario_actual );


    let cerrar_sesion_todos = warp::post()
	.and( warp::path("cerrar_sesion_todos") )
	.and( warp::header::value("Authorization") )
	.and( db_filter.clone() )
	.and_then( crate::handlers::usuarios::cerrar_sesion_todos );

    let ping = warp::path("ping")
	.map(|| "pong");    

    let api_1 = warp::path("api")
	.and( check_authenticated.clone() )
	.and( cerrar_sesion
	      .or( cerrar_sesion_todos )
	      .or( ping )
	      .or( usuarios_1 )
	      .or( usuarios_2 )
	      .with( warp::reply::with::header("Content-Type", "application/json; charset=utf-8") )
	);


    let api_2 = warp::path("api")
	.and( check_authenticated.clone() )
	.and( usuarios_3
	      .or( usuario_actual )
	      .or( entrada_cana )
	      .or( informe_produccion )
	      .or( liquidaciones_cana )
		  .or( pqrs_admin )
	      .with( warp::reply::with::header("Content-Type", "application/json; charset=utf-8") )
	);


    let api_3 = warp::path("api")
	.and( check_authenticated.clone() )
	.and( admin_documentos_1
	      .or( admin_documentos_2 )
	      .or( asistencia_tecnica )
	      .or( formato_vinculacion_natural )
	      .or( formato_vinculacion_juridica )
	      .with( warp::reply::with::header("Content-Type", "application/json; charset=utf-8") )
	);


    let api_4 = warp::path("api")
	.and( check_authenticated.clone() )
	.and( formato_autorizacion_tratamiento
	      .or( admin_documentos_3 )
	      .or( cronologico )
	      .or( roles_1 )
	      .or( roles_2 )
	      .with( warp::reply::with::header("Content-Type", "application/json; charset=utf-8") )
	);
   

	let api_5 = warp::path("api")
	.and( check_authenticated.clone() )
	.and( establecer_aceptacion_politica_admin_datos
	      .with( warp::reply::with::header("Content-Type", "application/json; charset=utf-8") )
	);
	
    
    
    let favicon = warp::path("favicon.ico")
	.and( warp::fs::file("./static/favico.ico") );

    let root = warp::get()
	.and( warp::path::end() )
	.map(|| { warp::redirect(Uri::from_static("/static/index.html")) } );

    let tls_challenge = warp::get()
	.and(warp::path(".well-known"))
	.and(warp::path("acme-challenge"))
	.and(warp::fs::dir("./tls"));

    let static_dir = warp::get()
	.and(warp::path("static"))
	.and(warp::fs::dir("./static"));

    let docs = warp::get()
	.and(warp::path("docs"))
	.and(warp::fs::dir("./doc/doc"));

    let tls_request = warp::get()
	.and(warp::path!("tls" / "order_cert"))
	.and(tls_cert_name_filter)
	.and(tls_cert_account_filter)
	.and_then(crate::handlers::tls::order_cert);

    let download = warp::get()
	.and(warp::path("download"))
	.and(warp::fs::dir("./tmp"));


    let file_upload = warp::put()
	.and(warp::body::content_length_limit(1024 * 1024 * 32))
	.and(warp::path("upload"))
	.and(warp::body::aggregate())
	.and_then(crate::handlers::file_upload_handler);

    let static_upload = warp::put()
	.and(warp::body::content_length_limit(1024 * 1024 * 32))
	.and(warp::path("static_upload"))
	.and(warp::body::aggregate())
	.and_then(crate::handlers::static_upload_handler);


    let static_upload = warp::put()
	.and(warp::body::content_length_limit(1024 * 1024 * 32))
	.and(warp::path("static_upload"))
	.and(warp::body::aggregate())
	.and_then(crate::handlers::static_upload_handler);


    let db_manager_listar = warp::path("listar")
	.and( warp::body::json() )
	.and( db_filter.clone() )
	.and_then(crate::handlers::db_manager::listar);

    let db_manager_borrar = warp::path("borrar")
	.and( warp::body::json() )
	.and( db_filter.clone() )
	.and_then(crate::handlers::db_manager::borrar);

    
    let db_manager = warp::post()
	.and( warp::path("db_manager") )
	.and( check_authenticated.clone() )
	.and( db_manager_listar
	      .or( db_manager_borrar )
	      .with( warp::reply::with::header("Content-Type", "application/json; charset=utf-8") )
	);

    
	      
    static_dir.boxed()
	.or( docs.boxed() ).boxed()
	.or( root.boxed() ).boxed()
	.or( api_1.boxed() ).boxed()
	.or( api_2.boxed() ).boxed()
	.or( api_3.boxed() ).boxed()
	.or( api_4.boxed() ).boxed()
	.or( api_5.boxed() ).boxed()
	.or( favicon.boxed() ).boxed()
	.or( tls_challenge.boxed() ).boxed()
	.or( tls_request.boxed() ).boxed()
	.or( file_upload.boxed() ).boxed()
	.or( static_upload.boxed()).boxed()
	.or( autenticar.boxed() ).boxed()
	.or( actualizar_token.boxed() ).boxed()
	.or( solicitar_recuperar_clave.boxed()  ).boxed()
	.or( recuperar_clave.boxed()  ).boxed()
	.or( download.boxed() ).boxed()
	.or( db_manager.boxed() ).boxed()
	.with(log)
        .recover(handle_rejection)

}

use std::error::Error;

async fn handle_rejection(err: Rejection) -> Result<impl Reply, std::convert::Infallible> {

    if let Some(response) = err.find::<crate::interfaces::ResponseStructure>(){
        
        Ok(warp::reply::json(&response))

    } else if let Some(e) = err.find::<warp::filters::body::BodyDeserializeError>() {
        // This error happens if the body could not be deserialized correctly
        // We can use the cause to analyze the error and customize the error message

        let message = match e.source() {
            Some(cause) => cause.to_string(),
            None => "No se puede encontrar un mensaje de error interno para mostrar".to_string(),
        };

        
        let json = crate::interfaces::ResponseStructure{
            response_status: crate::interfaces::ResponseStatus{
                code: "Z001".into(),
                message: message,
                traceback: "".into()    
            },
            payload: serde_json::Value::Null,
        };

        Ok(warp::reply::json(&json))

    } else{
        

        let json = crate::interfaces::ResponseStructure{
            response_status: crate::interfaces::ResponseStatus{
                code: "Z000".into(),
                message: format!("Solicitud no encontrada: {:?}",err),
                traceback: "".into()    
            },
            payload: serde_json::Value::Null,
        };

            
        Ok(warp::reply::json(&json))
    }
}
