use chrono::prelude::*;
use serde_derive::{Deserialize, Serialize};
use std::cmp::Ordering;
use crate::errors;
use super::{Auditable, Storable};
use crate::{audit,store};
use super::{
    sesion::Sesion,
};
audit!{Usuario}
store!{Usuario, usuario_id, vec![], vec![]}
use std::default::Default;

pub type id_rol = String;

#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct Usuario{
    usuario_id: String,
    usuario: String,
    nombre_completo: String,
    clave_acceso: String,
    telefono1: String,
    telefono2: String,
    telefono3: String,
    email: String,
    email_alternativo: String,
    nits: Vec<String>,
    reintentos_acceso: usize,
    bloqueo: bool,
    roles: Vec<id_rol>,
    creation_stamp: i64,
    created_by: String,
    modification_stamp: i64,
    modified_by: String,
    #[serde(default)]
    token_firebase: String, 
	#[serde(default)]
	aceptacion_politica_admin_datos:Option<i64>,
	#[serde(default)]
	eliminado:Option<i64>

}

impl std::default::Default for Usuario {
    fn default () -> Self {
	Self{
	    usuario_id: Self::generate_id(),
	    usuario: Default::default(),
	    nombre_completo: Default::default(),
	    clave_acceso: Default::default(),
	    telefono1: Default::default(),
	    telefono2: Default::default(),
	    telefono3: Default::default(),
	    email: Default::default(),
	    email_alternativo: Default::default(),
	    nits: Default::default(),
	    reintentos_acceso: Default::default(),
	    bloqueo: Default::default(),
	    roles: Default::default(),
	    creation_stamp: crate::utils::now_microseconds(),
	    created_by: Default::default(),
	    modification_stamp: crate::utils::now_microseconds(),
	    modified_by: Default::default(),
	    token_firebase: Default::default(),
		aceptacion_politica_admin_datos:None,
		eliminado:None,
	}
    }
}


impl Usuario {

    pub fn new(usuario:&str) -> Self{
	Self{
	    usuario: usuario.to_string(),
	    ..Default::default()
	}
    }
    
    pub fn crear(_db : &sled::Db, usuario:&str, clave_acceso: &str, nombre_completo:&str, telefono1: &str,telefono2: &str,telefono3: &str,email: &str,email_alternativo: &str,nits: &Vec<String>, roles: &Vec<id_rol>) -> Result<Self, Box<dyn std::error::Error>> {

	match Self::scan_first(&_db, |x| x.usuario() == usuario){
	    None => Ok(
		Self{
		    usuario_id: "".to_string(),
		    usuario: usuario.to_string(),
		    nombre_completo: nombre_completo.to_string(),
		    clave_acceso: Self::hash_clave_acceso(clave_acceso),
		    telefono1:telefono1.to_string(),
		    telefono2:telefono2.to_string(),
		    telefono3:telefono3.to_string(),
		    email:email.to_string(),
		    email_alternativo:email_alternativo.to_string(),
		    nits: nits.clone(),
		    reintentos_acceso: 0,
		    bloqueo: false,
		    roles: roles.clone(),
		    creation_stamp: 0,
		    created_by:"".to_string(),
		    modification_stamp: 0,
		    modified_by:"".to_string(),
		    token_firebase: Default::default(),
			aceptacion_politica_admin_datos:None,
			eliminado: None,
		}
	    ),
	    Some(usuario)=> Err(Box::new(errors::StorableError::DuplicateRecord( usuario.usuario(), usuario.id()) )),
	}
    }

    pub fn agregar_roles(&mut self, listado: &mut Vec<String>){
	self.roles.append(listado);
	self.roles.sort_by(|a,b| a.cmp(&b) );
	self.roles.dedup();
    }

    pub fn remover_roles(&mut self, listado: &Vec<String>){
	self.roles = self.roles.iter().filter_map(|x| if !listado.contains(x) { Some( x.to_string() ) } else { None } ).collect();
    }

    pub fn roles(&self) -> Vec<String>{
	self.roles.clone()
    }

    
    pub fn copiar_instancia(&self, encriptar_clave: bool) -> Self {
	let mut aux = self.clone();

	if encriptar_clave == true{
	    aux.clave_acceso = Self::hash_clave_acceso(&self.clave_acceso);
	}

	aux
    }

    pub fn actualizar_datos(&mut self, datos_a_actualizar: &Usuario){
	let mut aux = datos_a_actualizar.clone();

	aux.clave_acceso = self.clave_acceso.to_string();
        aux.roles = self.roles.clone();
	*self = aux;

    }

    pub fn check_before_store(&self, _db : &sled::Db) -> Result<(),Box<dyn std::error::Error>>{

	if let Some(user) = Self::scan_first(_db,|x| x.email() == self.email() ){
	    
	    Err(
		Box::new(
		    errors::StorableError::WriteError(
			format!(
			    "El usuario {} ya se encuentra vinculado al correo {}. No puede utilizarse para crear el usuario {}.",
			    user.usuario(),
			    user.email(),
			    self.usuario()
			)
		    )
		)
	    )
		
	}else{
	    Ok(())
	}
	
    }

    pub fn usuario_anonimo() -> Self {
	Self{
	    usuario_id: "".to_string(),
	    usuario: "".to_string(),
            nombre_completo: "".to_string(),
	    clave_acceso: "".to_string(),
            telefono1:"".to_string(),
            telefono2:"".to_string(),
            telefono3:"".to_string(),
            email:"".to_string(),
            email_alternativo:"".to_string(),
	    nits: vec![],
            reintentos_acceso: 0,
            bloqueo: false,
	    roles: vec![],
            creation_stamp: 0,
            created_by:"".to_string(),
            modification_stamp: 0,
            modified_by:"".to_string(),
	    token_firebase: Default::default(),
		aceptacion_politica_admin_datos:None,
		eliminado :None
	}
    }
	
	pub fn set_usuario(&mut self,i:&str){
		self.usuario = i.to_string();
	}

	pub fn set_nombre_completo(&mut self,i:&str){
		self.nombre_completo = i.to_string();
	}

	pub fn set_telefono1(&mut self,i:&str){
		self.telefono1 = i.to_string();
	}

	pub fn set_telefono2(&mut self,i:&str){
		self.telefono2 = i.to_string();
	}

	pub fn set_email(&mut self,i:&str){
		self.email= i.to_string();
	}

	pub fn set_bloqueo(&mut self,i:bool){
		self.bloqueo= i ;
	}
    
    pub fn token_firebase(&self) -> String{
	self.token_firebase.to_string()
    }


    pub fn set_reintentos_acceso(&mut self,i:usize){
	self.reintentos_acceso = i
    }
    
    pub fn reintentos_acceso(&self)-> usize{
	self.reintentos_acceso
    }

    pub fn usuario(&self) -> String{
	self.usuario.to_string()
    }

    pub fn bloqueo(&self) -> bool{
	self.bloqueo
    }

    pub fn email(&self) -> String{
	self.email.to_string()
    }

	pub fn eliminado(&self) -> Option<i64>{
		self.eliminado
	}

	pub fn set_eliminado(&mut self, i:i64){
		self.eliminado = Some(i);
	} 

	pub fn aceptacion_politica_admin_datos(&self) -> Option<i64>{
		self.aceptacion_politica_admin_datos
	}

	pub fn set_aceptacion_politica_admin_datos(&mut self, i:i64){
		self.aceptacion_politica_admin_datos = Some(i);
	} 

	pub fn clear_aceptacion_politica_admin_datos(&mut self){
		self.aceptacion_politica_admin_datos = None;
	}

    fn hash_clave_acceso(clave_acceso: &str) -> String{
	let mut hasher = crypto::sha1::Sha1::new();
	
	hasher.input_str(clave_acceso);

        use crypto::digest::Digest;

        hasher.result_str().to_string()
    }

    pub fn set_clave_acceso(&mut self, clave_acceso: &str, sesion: &Sesion){
	
	self.clave_acceso = Self::hash_clave_acceso(clave_acceso);
        self.record_modification( sesion, Utc::now().timestamp() );
    }

    pub fn establecer_clave_vacia(&mut self ){
	self.clave_acceso = "".to_string();
    }

    pub fn verificar_clave_acceso(&self, clave_acceso: &str) -> Result<(), errors::AuthenticationError>{
	if self.clave_acceso == Self::hash_clave_acceso(clave_acceso){
	    Ok( () )
	} else {
	    Err(errors::AuthenticationError::InvalidCredentials)
	}
    }

    pub fn nombre_completo(&self)->String {
	self.nombre_completo.to_string()
    }

    pub fn telefono1(&self)->String {
	self.telefono1.to_string()
    }

	pub fn telefono2(&self)->String {
	self.telefono2.to_string()
	}

    pub fn nits(&self)->Vec<String> {
	self.nits.clone()
    }

    pub fn refresh_instance(&mut self, _db : &sled::Db) {

	if let Ok(Some(aux)) = Self::load(_db, &self.id()){
	    *self = aux;
	}
    }

    pub fn set_nits(&mut self, o: &Vec<String> ) {
	self.nits = o.clone();
    }
    
    pub fn set_token_firebase(&mut self, i:&str){
	self.token_firebase = i.to_string();
    }

	pub fn usuario_id(&self)-> String{
        self.usuario_id.to_string()
    }

}
