use crate::{tmpzipfmt,tmppdffmt};
use chrono::prelude::*;
use serde_derive::{Deserialize, Serialize};
use crate::errors;
use super::{
    usuario::Usuario,
    sesion::Sesion,
    Storable,
};
use crate::{audit, store};
use std::default::Default;
use super::documento::{
    TipoDocProcesado,
    Documento,
};
audit!{OrdenProcesamientoDocumentos}
store!{OrdenProcesamientoDocumentos, id, vec![], vec![]}

#[derive(Deserialize, Serialize, Debug, Clone)]
pub enum EstadoDistribucion{
    Creado(i64),
    FuenteCargada(i64),
    Procesamiento(i64),
    Procesado(i64),
    Distribuido(i64),
    Eliminado(i64),
}

impl Default for EstadoDistribucion{
    fn default() -> Self{
	let now = crate::utils::now_microseconds();
	Self::Creado(now)
    }
}

impl std::fmt::Display for EstadoDistribucion {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
	let (text, stamp) = match self {
	    EstadoDistribucion::Creado(stamp) => ("Creado en",stamp),
	    EstadoDistribucion::FuenteCargada(stamp) => ("Fuente cargada en",stamp),
	    EstadoDistribucion::Procesamiento(stamp) => ("Procesamiento desde",stamp),
	    EstadoDistribucion::Procesado(stamp) => ("Procesado en",stamp),
	    EstadoDistribucion::Distribuido(stamp) => ("Distribuido en",stamp),
	    EstadoDistribucion::Eliminado(stamp) => ("Eliminado en",stamp),
	};

	let dt = Utc.timestamp(crate::utils::convert_microseconds_unix_timestamp(*stamp), 0u32).with_timezone(&FixedOffset::west(3600*5));
	
        write!(f, "{} : {}", text, dt)
    }
}

pub enum ItemProcesado{
    CorrectamenteProcesado{ruta_pagina: std::path::PathBuf, llaves: Vec<String>, numero_pagina: String},
    ErrorProcesando{ruta_pagina: std::path::PathBuf, numero_pagina: String, motivo: String},
}

struct PaginaError{
    pagina: String,
    motivo: String,
}

#[derive(Deserialize, Serialize, Debug, Clone)]
pub struct OrdenProcesamientoDocumentos{
    id: String,
    tipo:TipoDocProcesado,
    estado: EstadoDistribucion,
    historial_estado: Vec<EstadoDistribucion>,
    notas: String,
	#[serde(default)]
	preliminar: bool,
    creation_stamp: i64,
    created_by: String,
    modification_stamp: i64,
    modified_by: String,
    nombre_archivo: String,
}

impl std::default::Default for OrdenProcesamientoDocumentos {
    fn default () -> Self {
    let now = crate::utils::now_microseconds();
	let aux_estado : EstadoDistribucion = Default::default();
	let aux_historial_estado = vec![aux_estado.clone()];
	
	Self{
	    id: Default::default(),
	    tipo: Default::default(),
	    estado: aux_estado,
	    historial_estado: aux_historial_estado,
	    notas: Default::default(),
		preliminar: Default::default(),
	    creation_stamp: now,
	    created_by: Default::default(),
	    modification_stamp: now,
	    modified_by: Default::default(),
		nombre_archivo: Default::default(),
	}
    }
}

impl OrdenProcesamientoDocumentos {
    pub fn new() -> Self {
	Default::default()
    }

    pub fn tipo(&self) -> TipoDocProcesado {
	self.tipo.clone()
    }

    pub fn estado(&self) -> EstadoDistribucion {
	self.estado.clone()
    }

	pub fn preliminar(&self) -> bool {
		 self.preliminar
	}
	pub fn set_preliminar(&mut self,i:bool) {
		self.preliminar= i;
	}
    
    pub fn notas(&self) -> String {
	self.notas.clone()
    }

    pub fn historial_estado(&self) -> Vec<EstadoDistribucion> {
	self.historial_estado.clone()
    }

    pub fn fuente(&self, db: &std::sync::Arc<sled::Db>) -> Result<sled::IVec, Box<std::error::Error>> {
	crate::utils::leer_blob(db, &self.id)
    }

    pub fn set_tipo(&mut self, i: &TipoDocProcesado){
	self.tipo = i.clone();
    }

    pub fn set_notas(&mut self, i: String){
	self.notas = i;
    }
    
    pub fn set_estado(&mut self, i: EstadoDistribucion) {
	self.estado = i.clone();
	self.historial_estado.push(i);
    }

    pub fn set_historial_estado(&mut self, i: &Vec<EstadoDistribucion>){
	self.historial_estado = i.clone();
    }

    pub fn set_fuente(&mut self, db: &std::sync::Arc<sled::Db>,i: Vec<u8>) -> Result<(), Box<dyn std::error::Error>>{
	crate::utils::guardar_blob(db, &self.id, i)
    }

	pub fn set_nombre_archivo(&mut self, i: &str ){
		self.nombre_archivo = i.to_string();
	}

	pub fn nombre_archivo( &self ) -> String { 
		self.nombre_archivo.to_string()
	}

    pub fn act_est_proc(id: String, estado_procesos_mutex: &std::sync::Arc<std::sync::Mutex<std::collections::HashMap<String,String>>>, i: String) -> Result<(), Box<dyn std::error::Error>>{
	
	let mut estado_procesos = estado_procesos_mutex
	    .lock()
	    .map_err(|e| Box::new(errors::ProcesamientoDocumentos::EstadoOrdenesNoSePuedeBloquear{motivo_interno: e.to_string()}) )?;
	
	estado_procesos.insert(id, i);
	drop(estado_procesos);
	Ok(())
	
    }


    pub fn consultar_est_proc(id: String, estado_procesos_mutex: &std::sync::Arc<std::sync::Mutex<std::collections::HashMap<String,String>>>) -> Result<String, Box<dyn std::error::Error>>{
	
	let mut estado_procesos = estado_procesos_mutex
	    .lock()
	    .map_err(|e| Box::new(errors::ProcesamientoDocumentos::EstadoOrdenesNoSePuedeBloquear{motivo_interno: e.to_string()}) )?;
	
	let out = estado_procesos
	    .get(&id)
	    .unwrap_or(&format!("No se encuentra la orden: {}",id))
	    .to_string();
	
	drop(estado_procesos);
	
	Ok(out.to_string())
	
    }

    pub fn eliminar_est_proc(id: String, estado_procesos_mutex: &std::sync::Arc<std::sync::Mutex<std::collections::HashMap<String,String>>>) -> Result<(), Box<dyn std::error::Error>>{

        let mut estado_procesos = estado_procesos_mutex
	    .lock()
	    .map_err(|e| Box::new(errors::ProcesamientoDocumentos::EstadoOrdenesNoSePuedeBloquear{motivo_interno: e.to_string()}))?;
	
	estado_procesos.remove(&id);
	drop(estado_procesos);

	Ok(())

    }


    
    pub fn procesar(&mut self, sesion: &Sesion, db: &std::sync::Arc<sled::Db>, estado_procesos_mutex: &std::sync::Arc<std::sync::Mutex<std::collections::HashMap<String,String>>>) -> Result<(), Box<dyn std::error::Error>>{

	let now = crate::utils::now_microseconds();
	self.set_estado(EstadoDistribucion::Procesamiento(now));

	for mut doc in Documento::scan(&db, |x| x.orden_procesamiento_documentos_id() == self.id() ).iter_mut(){
	    doc.delete(&db);
	    doc.delete_data(&db)?;
	}
	
	self.store(&db, &sesion)
	    .map_err(|e| errors::ProcesamientoDocumentos::FallaDistribucion("no se puede guardar cambio de estado iniciando proceso.".to_string(), e.to_string()))?;

	Self::act_est_proc(self.id(),estado_procesos_mutex,"Iniciando...".to_string())?;
	
	match self.tipo{
	    TipoDocProcesado::DonacionesCenicana => self.procesar_DonacionesCenicana(&sesion, &db),
	    TipoDocProcesado::FondoSocial => self.procesar_FondoSocial(&sesion, &db),
	    TipoDocProcesado::Retenciones => self.procesar_Retenciones(&sesion, &db),
	    TipoDocProcesado::IngresosYCostos => self.procesar_IngresosYCostos(&sesion, &db),
	    TipoDocProcesado::Ica => self.procesar_Ica(&sesion, &db),
	    TipoDocProcesado::LiquidacionCana => self.procesar_LiquidacionCana(&sesion, &db, estado_procesos_mutex),
	    TipoDocProcesado::LiquidacionMercadoExcedentario => self.procesar_LiquidacionMercadoExcedentario(&sesion, &db, estado_procesos_mutex),
	    TipoDocProcesado::LiquidacionAnticipos => self.procesar_LiquidacionAnticipos(&sesion, &db, estado_procesos_mutex),
	    TipoDocProcesado::Ninguno => self.procesar_Ninguno(),
		TipoDocProcesado::LiquidacionCanaParticipacion => self.procesar_LiquidacionCanaParticipacion(&sesion, &db, estado_procesos_mutex),
	}?;

	Self::act_est_proc(self.id(),estado_procesos_mutex,"Finalizando...".to_string())?;

	self.set_estado(EstadoDistribucion::Procesado(now));

	self.store(&db, &sesion)
	    .map_err(|e| errors::ProcesamientoDocumentos::FallaDistribucion("no se puede guardar cambio de estado a finalizado proceso.".to_string(), e.to_string()))?;

	Ok(())
	
    }

    fn procesar_zip<T>(&mut self, sesion: &Sesion, db: &std::sync::Arc<sled::Db>, tipo:TipoDocProcesado, generador_llaves: T) -> Result<(),Box<dyn std::error::Error>>
    where
	T: Fn(String) -> Result<Vec<String>, Box<dyn std::error::Error>>
    {
	let zip_file_name_path = tmpzipfmt!(crate::utils::generate_id());
	let fuente = self.fuente(db)?;
	std::fs::write(&zip_file_name_path, &fuente)
	    .map_err(|e| errors::ProcesamientoDocumentos::ImposibleEscribirArchivoTemporal{nombre_archivo: zip_file_name_path.to_string(), motivo_interno: e.to_string()})?;


	let zip_file_path = std::path::Path::new(&zip_file_name_path);
	let file = std::fs::File::open(&zip_file_path)
	    .map_err(|e| errors::ProcesamientoDocumentos::FallaDescomprimiendoArchivo( zip_file_name_path.to_string(), e.to_string() ))?;

	let mut archive = zip::ZipArchive::new(file)
			       .map_err(|e| errors::ProcesamientoDocumentos::FallaDescomprimiendoArchivo( zip_file_name_path.to_string(), e.to_string() ))?;

	for i in 0..archive.len() {
	    
            let mut file = archive.by_index(i)
		.map_err(|e| errors::ProcesamientoDocumentos::FallaDescomprimiendoArchivo( zip_file_name_path.to_string(), e.to_string() ))?;
	    
	    let mut enclosed_filename = file
		.enclosed_name()
		.ok_or(
		    errors::ProcesamientoDocumentos::FallaDescomprimiendoArchivo(
			zip_file_name_path.to_string(),
			format!("No es posible resolver el nombre para el archivo numero {} del archivo comprimido (A)",i)
		    )
		)?
		.to_str()
		.ok_or(
		    errors::ProcesamientoDocumentos::FallaDescomprimiendoArchivo(
			zip_file_name_path.to_string(),
			format!("No es posible resolver el nombre para el archivo numero {} del archivo comprimido (B)",i)
		    )
		)?.to_string();

	    use std::io::Read;
	    let mut data = vec![];

	    file.read_to_end(&mut data);

	    let mut doc = Documento::new();

	    let llaves : Vec<String> = generador_llaves(enclosed_filename.to_string())?;
	    doc.set_llaves(llaves.clone());
	        
	    doc.set_tipo(tipo.clone());

	    doc.set_orden_procesamiento_documentos_id(&self.id());
        
	    let fecha_documento = DateTime::parse_from_str(&format!("{}-01-01 08:00:00 -05:00",dbg!(&llaves[1])), "%Y-%m-%d %H:%M:%S %z").ok().map(|x| crate::utils::convert_unix_timestamp_microseconds(x.timestamp()) );

	    doc.set_fecha_documento(fecha_documento);
		dbg!(&fecha_documento);
        dbg!(&doc);
	    doc.store(db, sesion)?;

	    dbg!(doc).set_data(&db, data)?;
	    
	    
	}

	Ok(())

    }

    fn procesar_pdf_multipagina<T>(&mut self,
				   sesion: &Sesion,
				   db: &std::sync::Arc<sled::Db>,
				   tipo:TipoDocProcesado,
				   estado_procesos_mutex: &std::sync::Arc<std::sync::Mutex<std::collections::HashMap<String,String>>>,
				   generador_lista_documentos: T)
				   -> Result<(),Box<dyn std::error::Error>>
    where
	T: Fn(Vec<std::path::PathBuf>) -> Result<Vec<ItemProcesado>, Box<dyn std::error::Error>>
    {
	//Escribir el archivo fuente en unra ruta temporal
	let ruta_archivo_pdf = tmppdffmt!(crate::utils::generate_id());
	let path_archivo_pdf = std::path::Path::new(&ruta_archivo_pdf);
	let fuente = self.fuente(db)?;
	
	std::fs::write(&path_archivo_pdf, &fuente)
	    .map_err(|e| errors::ProcesamientoDocumentos::ImposibleEscribirArchivoTemporal{nombre_archivo: ruta_archivo_pdf.to_string(), motivo_interno: e.to_string()})?;

	//Separar el archivo fuente en archivos por cada pagina
	let rutas_paginas = crate::utils::separar_pdf_en_paginas(&path_archivo_pdf)?;

	//Crear una cola de documentos a generar con sus respectivas llaves
	let mut cola_documentos_a_generar : Vec<ItemProcesado> = generador_lista_documentos(rutas_paginas)?;


	let total_paginas = cola_documentos_a_generar.len();
	let mut paginas_error : Vec<PaginaError> = vec![];

	for (i,item) in cola_documentos_a_generar.iter().enumerate(){

	    match item{
		ItemProcesado::CorrectamenteProcesado{ruta_pagina: ruta_pagina, llaves: llaves, numero_pagina: numero_pagina} => {
		    
		    Self::act_est_proc(self.id(),estado_procesos_mutex,format!("Almacenando en DB pagina {} de {}",i,total_paginas))?;

		    let mut doc = Documento::new();
		    
		    doc.set_llaves(llaves.clone());

		    let fecha_documento;
		    
		    if tipo == TipoDocProcesado::LiquidacionMercadoExcedentario{
			fecha_documento = DateTime::parse_from_str(&format!("{}-01-01 00:00:00 -05:00", llaves[1]), "%Y-%m-%d %H:%M:%S %z").ok().map(|x| crate::utils::convert_unix_timestamp_microseconds(x.timestamp()) );
		    } else {
			fecha_documento = DateTime::parse_from_str(&format!("{} 00:00:00 -05:00", llaves[1]), "%Y-%m-%d %H:%M:%S %z").ok().map(|x| crate::utils::convert_unix_timestamp_microseconds(x.timestamp()) );
		    }
		    doc.set_fecha_documento(fecha_documento);
		    
		    doc.set_tipo(TipoDocProcesado::LiquidacionCana);

		    let data = std::fs::read(ruta_pagina.as_path()).
			map_err(|e| errors::ProcesamientoDocumentos::ArchivoFuenteNoEncontrado{nombre_archivo: ruta_pagina.to_string_lossy().to_string()})?;

		    doc.set_orden_procesamiento_documentos_id(&self.id());

		    match doc.store(db, sesion){
			Ok(_)=>(),
			Err(why) => {
			    paginas_error.push(
				PaginaError{
				    pagina: numero_pagina.to_string(),
				    motivo: "Error al intentar almacenar el documento en base de datos".to_string(),
				}
			    );
			    continue;
			}
		    }

		    doc.set_data(&db, data)?;

		    match std::fs::remove_file(ruta_pagina.as_path()){
			Ok(_)=>(),
			Err(why) => {
			    continue;
			}
		    }
		},
		ItemProcesado::ErrorProcesando{ruta_pagina: ruta_pagina, numero_pagina: numero_pagina, motivo: motivo} => {

		    match std::fs::remove_file(ruta_pagina.as_path()){
			Ok(_)=>(),
			Err(why) => {
			    paginas_error.push(
				PaginaError{
				    pagina: numero_pagina.to_string(),
				    motivo: format!("Error al intentar borrar el el pdf especifico para la pagina {:?}",ruta_pagina),
				}
			    );
			    continue;
			}
		    }
		    
		    paginas_error.push(
			PaginaError{
			    pagina: numero_pagina.to_string(),
			    motivo: motivo.to_string(),
			}
		    );
		},
	    }

	}

	
	match std::fs::remove_file(path_archivo_pdf){
	    Ok(_)=>(),
	    Err(why) => (),
	}

	if paginas_error.len() > 0 {

	    let paginas_error_string = paginas_error.iter().map(|item|{
		format!("Página: {}, Problema: {}",item.pagina, item.motivo)
	    }).collect::<Vec<String>>().join("),(");
	    
	    return Err(Box::new(errors::ProcesamientoDocumentos::FallaLecturaCampos{numeros_paginas: format!("({})",paginas_error_string) }));
	}

	Ok(())
	    
    }


    fn procesar_DonacionesCenicana(&mut self, sesion: &Sesion, db: &std::sync::Arc<sled::Db>) -> Result<(), Box<dyn std::error::Error>>{

	self.procesar_zip(sesion, db, TipoDocProcesado::DonacionesCenicana, |archivo_encontrado|{

	    let pathbuf_archivo = std::path::Path::new(&archivo_encontrado).to_path_buf();
	    let nombre = pathbuf_archivo.file_stem()
		.ok_or(Box::new(errors::Converciones::FalloInterpretandoNombreArchivo{origen: pathbuf_archivo.to_string_lossy().to_string()}))?
		.to_str()
		.ok_or(Box::new(errors::Converciones::FalloInterpretandoNombreArchivo{origen: pathbuf_archivo.to_string_lossy().to_string()}))?;
	    
	    let llaves : Vec<String> = nombre.to_string().split("_").map(|x| x.to_string() ).collect();

	    if llaves.len() != 2{
		return Err(Box::new(errors::ProcesamientoDocumentos::FalloGenerandoLlaveDocumento{origen: nombre.to_string(), especificacion_llaves: "'XXXXXXX_YYYY', donde XXXXX nit y YYYY año".to_string()}))
	    }
	    
	    Ok(llaves)
		
	})?;
	
	Ok(())
    }

    fn procesar_FondoSocial(&mut self, sesion: &Sesion, db: &std::sync::Arc<sled::Db>) -> Result<(), Box<dyn std::error::Error>>{

	self.procesar_zip(sesion, db, TipoDocProcesado::FondoSocial, |archivo_encontrado|{

	    let pathbuf_archivo = std::path::Path::new(&archivo_encontrado).to_path_buf();
	    let nombre = pathbuf_archivo.file_stem()
		.ok_or(Box::new(errors::Converciones::FalloInterpretandoNombreArchivo{origen: pathbuf_archivo.to_string_lossy().to_string()}))?
		.to_str()
		.ok_or(Box::new(errors::Converciones::FalloInterpretandoNombreArchivo{origen: pathbuf_archivo.to_string_lossy().to_string()}))?;
	    
	    let llaves : Vec<String> = nombre.to_string().split("_").map(|x| x.to_string() ).collect();

	    if llaves.len() != 2{
		return Err(Box::new(errors::ProcesamientoDocumentos::FalloGenerandoLlaveDocumento{origen: nombre.to_string(), especificacion_llaves: "'XXXXXXX_YYYY', donde XXXXX nit y YYYY año".to_string()}))
	    }
	    
	    Ok(llaves)
		
	})?;


	Ok(())
    }

    fn procesar_Retenciones(&mut self, sesion: &Sesion, db: &std::sync::Arc<sled::Db>) -> Result<(), Box<dyn std::error::Error>>{

	self.procesar_zip(sesion, db, TipoDocProcesado::Retenciones, |archivo_encontrado|{

	    let pathbuf_archivo = std::path::Path::new(&archivo_encontrado).to_path_buf();
	    let nombre = pathbuf_archivo.file_stem()
		.ok_or(Box::new(errors::Converciones::FalloInterpretandoNombreArchivo{origen: pathbuf_archivo.to_string_lossy().to_string()}))?
		.to_str()
		.ok_or(Box::new(errors::Converciones::FalloInterpretandoNombreArchivo{origen: pathbuf_archivo.to_string_lossy().to_string()}))?;
	    
	    let llaves : Vec<String> = nombre.to_string().split("_").map(|x| x.to_string() ).collect();

	    if llaves.len() != 2{
		return Err(Box::new(errors::ProcesamientoDocumentos::FalloGenerandoLlaveDocumento{origen: nombre.to_string(), especificacion_llaves: "'XXXXXXX_YYYY', donde XXXXX nit y YYYY año".to_string()}))
	    }
	    
	    Ok(llaves)
		
	})?;


	Ok(())
    }

    fn procesar_IngresosYCostos(&mut self, sesion: &Sesion, db: &std::sync::Arc<sled::Db>) -> Result<(), Box<dyn std::error::Error>>{

	self.procesar_zip(sesion, db, TipoDocProcesado::IngresosYCostos, |archivo_encontrado|{

	    let pathbuf_archivo = std::path::Path::new(&archivo_encontrado).to_path_buf();
	    let nombre = pathbuf_archivo.file_stem()
		.ok_or(Box::new(errors::Converciones::FalloInterpretandoNombreArchivo{origen: pathbuf_archivo.to_string_lossy().to_string()}))?
		.to_str()
		.ok_or(Box::new(errors::Converciones::FalloInterpretandoNombreArchivo{origen: pathbuf_archivo.to_string_lossy().to_string()}))?;
	    
	    let llaves : Vec<String> = nombre.to_string().split("_").map(|x| x.to_string() ).collect();

	    if llaves.len() != 2{
		return Err(Box::new(errors::ProcesamientoDocumentos::FalloGenerandoLlaveDocumento{origen: nombre.to_string(), especificacion_llaves: "'XXXXXXX_YYYY', donde XXXXX nit y YYYY año".to_string()}))
	    }
	    
	    Ok(llaves)
		
	})?;


	Ok(())
    }

    fn procesar_Ica(&mut self, sesion: &Sesion, db: &std::sync::Arc<sled::Db>) -> Result<(), Box<dyn std::error::Error>>{

	self.procesar_zip(sesion, db, TipoDocProcesado::Ica, |archivo_encontrado|{

	    let pathbuf_archivo = std::path::Path::new(&archivo_encontrado).to_path_buf();
	    let nombre = pathbuf_archivo.file_stem()
		.ok_or(Box::new(errors::Converciones::FalloInterpretandoNombreArchivo{origen: pathbuf_archivo.to_string_lossy().to_string()}))?
		.to_str()
		.ok_or(Box::new(errors::Converciones::FalloInterpretandoNombreArchivo{origen: pathbuf_archivo.to_string_lossy().to_string()}))?;
	    
	    let llaves : Vec<String> = nombre.to_string().split("_").map(|x| x.to_string() ).collect();

	    if llaves.len() != 2{
		return Err(Box::new(errors::ProcesamientoDocumentos::FalloGenerandoLlaveDocumento{origen: nombre.to_string(), especificacion_llaves: "'XXXXXXX_YYYY', donde XXXXX nit y YYYY año".to_string()}))
	    }
	    
	    Ok(llaves)
		
	})?;


	Ok(())
    }


    fn procesar_LiquidacionCana(&mut self, sesion: &Sesion, db: &std::sync::Arc<sled::Db>, estado_procesos_mutex: &std::sync::Arc<std::sync::Mutex<std::collections::HashMap<String,String>>>) -> Result<(), Box<dyn std::error::Error>>{
	
	Self::act_est_proc(self.id(),estado_procesos_mutex,format!("Separando PDF en paginas..."))?;

	let id = self.id();
	
	self.procesar_pdf_multipagina(sesion, db, TipoDocProcesado::LiquidacionCana, estado_procesos_mutex, |mut rutas_paginas| {

	    let total_paginas = rutas_paginas.len();
	    
	    //Compilar las expresiones regulares para encontrlet re_nit = regex::Regex::new(r#"Nit \:[ ]{4,}(\d{8,})"#).expect("La expresion regular de nit liquidacion de cana esta mala");ar en cada pagina
	    //los campos que identifican cada liquidacion
	    let re_fecha = regex::Regex::new(r#"(\d{2})/(\d{2})/(\d{4})"#).expect("La expresion regular de fechas liquidacion de cana esta mala");
		let re_fecha_aux = regex::Regex::new(r#"(\d{2})/(\d{1})/(\d{4})"#).expect("La expresion regular de fechas liquidacion de cana esta mala");
	    let re_nit = regex::Regex::new(r#"Nit :[ ]{4,}(\d{7,})"#).expect("La expresion regular de nit liquidacion de cana esta mala");
	    let re_hacienda = regex::Regex::new(r#"Hacienda :[ ]{4,}(\d{3,})"#).expect("La expresion regular de hacienda liquidacion de cana esta mala");
	    let re_cuenta_proveedor = regex::Regex::new(r#"Cuenta Proveedor :[ ]{4,}(\d{3,})"#).expect("La expresion regular de cuenta proveedor liquidacion de cana esta mala");
	    let mut cola_documentos_a_generar : Vec<ItemProcesado> = vec![];

	    
	    //Iterar por todas pdf las paginas
	    for (i, ruta_pagina) in rutas_paginas.iter().enumerate() {

		Self::act_est_proc(id.clone(),estado_procesos_mutex,format!("Procesando campos pagina {} de {}",i,total_paginas))?;

		//Obtener el texto del pdf de la pagina
		let contenido_pagina = crate::utils::convertir_pdf_en_string( ruta_pagina.as_path() )?;
       //  std::fs::write("/home/eramirez/portal_proveedores_carmelita_backend/salida.txt",&contenido_pagina); 
		//Encontrar la fecha
		//      30/11/2020
		let mut fecha : Vec<String> = Default::default();
		for cap in re_fecha.captures_iter(&contenido_pagina) {
		    fecha.push(format!("{}-{}-{}",cap[3].to_string() ,cap[2].to_string(),cap[1].to_string()));
		}
		//Encontrar la fecha  si el codigo de linea 578 falla este es para los mes que no tiene 2 digitos
		//      30/1/2020
		if fecha.len() != 1 {
			for cap in re_fecha_aux.captures_iter(&contenido_pagina) {

				let fech_cap: String = cap[2].to_string();
    			let adj: String = "0".to_string();
				let fec_  =  format!("{}{}", adj, fech_cap);

				fecha.push(format!("{}-{}-{}",cap[3].to_string() , fec_ ,cap[1].to_string()));
			}
		}

		//Encontrar NIT
		//Nit :                                               8913002331
		let mut nit : Vec<String> = Default::default();
		for cap in re_nit.captures_iter(&contenido_pagina) {
		    nit.push(cap[1].to_string());
		}

		//Encontrar Hacienda
		//Hacienda :              300
		let mut hacienda : Vec<String> = Default::default();
		for cap in re_hacienda.captures_iter(&contenido_pagina) {
		    hacienda.push(cap[1].to_string());
		}

		//Encontrar cuenta proveedor
		//Cuenta Proveedor :                                  2000800
		let mut cuenta_proveedor : Vec<String> = Default::default();
		for cap in re_cuenta_proveedor.captures_iter(&contenido_pagina) {
		    cuenta_proveedor.push(cap[1].to_string());
		}

		//Si encuentra solo una instancia de todos los campos
		//agregar al listado a generar; en caso contrario fallar
		//indicando el numero de pagina y el campo
		let numero_pagina = if let Some(stem) = ruta_pagina.file_stem() {
		    stem
			.to_string_lossy()
			.to_string()
			.split("_")
			.nth(1)
			.unwrap_or("(numero pagina no se pudo determinar)")
			.to_string()
		} else {
		    "(numero pagina no se pudo determinar)".to_string()
		};
		
		if fecha.len() == 1 && nit.len() == 1 && hacienda.len() == 1 && cuenta_proveedor.len() == 1{

		    cola_documentos_a_generar.push(
			ItemProcesado::CorrectamenteProcesado{
			    ruta_pagina: ruta_pagina.clone(),
			    llaves: vec![
				nit[0].to_string(),
				fecha[0].to_string(),
				hacienda[0].to_string(),
				cuenta_proveedor[0].to_string()
			    ],
			    numero_pagina: numero_pagina,
			}
		    );
		    
		}else{

		    let mut motivos :Vec<&str> = vec![];

		    if fecha.len() != 1 { motivos.push("fecha no encontrada"); }
		    if nit.len() != 1 { motivos.push("nit no encontrado"); }
		    if cuenta_proveedor.len() != 1 { motivos.push("cuenta proveedor no encontrada"); }
		    
		    cola_documentos_a_generar.push(
			ItemProcesado::ErrorProcesando{
			    ruta_pagina: ruta_pagina.clone(),
			    numero_pagina: numero_pagina,
			    motivo: motivos.join(",")
			}
		    );
		}
		
	    }

	    Ok(cola_documentos_a_generar)

	})

    }

    fn procesar_LiquidacionMercadoExcedentario(&mut self, sesion: &Sesion, db: &std::sync::Arc<sled::Db>, estado_procesos_mutex: &std::sync::Arc<std::sync::Mutex<std::collections::HashMap<String,String>>>) -> Result<(), Box<dyn std::error::Error>>{
	
	Self::act_est_proc(self.id(),estado_procesos_mutex,format!("Separando PDF en paginas..."))?;

	let id = self.id();
	
	self.procesar_pdf_multipagina(sesion, db, TipoDocProcesado::LiquidacionMercadoExcedentario, estado_procesos_mutex, |mut rutas_paginas| {

	    let total_paginas = rutas_paginas.len();
	    
	    //Compilar las expresiones regulares para encontrlet re_nit = regex::Regex::new(r#"Nit \:[ ]{4,}(\d{8,})"#).expect("La expresion regular de nit liquidacion de cana esta mala");ar en cada pagina
	    //los campos que identifican cada liquidacion
	    let re_fecha = regex::Regex::new(r#"Fecha:[ ]{4,}(\d{2})/(\d{2})/(\d{4})"#).expect("La expresion regular de fechas liquidacion de cana esta mala");
	    let re_nit = regex::Regex::new(r#"NIT :[ ]{4,}(\d{7,})"#).expect("La expresion regular de nit liquidacion de cana esta mala");
	    let re_hacienda = regex::Regex::new(r#"Hacienda:[ ]{4,}(\d{3,})"#).expect("La expresion regular de hacienda liquidacion de cana esta mala");
	    let re_cuenta_proveedor = regex::Regex::new(r#"Cuenta:[ ]{4,}(\d{3,})"#).expect("La expresion regular de cuenta proveedor liquidacion de cana esta mala");
	    let mut cola_documentos_a_generar : Vec<ItemProcesado> = vec![];

	    
	    //Iterar por todas pdf las paginas
	    for (i, ruta_pagina) in rutas_paginas.iter().enumerate() {

		Self::act_est_proc(id.clone(),estado_procesos_mutex,format!("Procesando campos pagina {} de {}",i,total_paginas))?;

		//Obtener el texto del pdf de la pagina
		let contenido_pagina = crate::utils::convertir_pdf_en_string( ruta_pagina.as_path() )?;

		//Encontrar Año
		//Fecha:                               31/12/2020
		let mut fecha : Vec<String> = Default::default();
		for cap in re_fecha.captures_iter(&contenido_pagina) {
		    fecha.push(format!("{}",cap[3].to_string()));
		}

		//Encontrar NIT
		//NIT :                                 8903009516
		let mut nit : Vec<String> = Default::default();
		for cap in re_nit.captures_iter(&contenido_pagina) {
		    nit.push(cap[1].to_string());
		}

		//Encontrar Hacienda
		//Hacienda:                              050              ESPERANZA MUNCHMEYER
		let mut hacienda : Vec<String> = Default::default();
		for cap in re_hacienda.captures_iter(&contenido_pagina) {
		    hacienda.push(cap[1].to_string());
		}

		//Encontrar cuenta proveedor
		//Cuenta:                               2000810
		let mut cuenta_proveedor : Vec<String> = Default::default();
		for cap in re_cuenta_proveedor.captures_iter(&contenido_pagina) {
		    cuenta_proveedor.push(cap[1].to_string());
		}

		//Si encuentra solo una instancia de todos los campos
		//agregar al listado a generar; en caso contrario fallar
		//indicando el numero de pagina y el campo
		let numero_pagina = if let Some(stem) = ruta_pagina.file_stem() { stem.to_string_lossy().to_string().split("_").nth(1).unwrap_or("(numero pagina no se pudo determinar)").to_string() } else { "(numero pagina no se pudo determinar)".to_string() };
		if fecha.len() == 1 && nit.len() == 1 && hacienda.len() == 1 && cuenta_proveedor.len() == 1{

		    cola_documentos_a_generar.push(
			ItemProcesado::CorrectamenteProcesado{
			    ruta_pagina: ruta_pagina.clone(),
			    llaves: vec![
				nit[0].to_string(),
				fecha[0].to_string(),
				hacienda[0].to_string(),
				cuenta_proveedor[0].to_string()
			    ],
			    numero_pagina: numero_pagina,
			}
		    );

		    
		}else{
		    
		    let mut motivos :Vec<&str> = vec![];

		    if fecha.len() != 1 { motivos.push("fecha no encontrada"); }
		    if nit.len() != 1 { motivos.push("nit no encontrado"); }
		    if cuenta_proveedor.len() != 1 { motivos.push("cuenta proveedor no encontrada"); }
		    if hacienda.len() != 1 { motivos.push("hacienda no encontrada"); }
		    
		    cola_documentos_a_generar.push(
			ItemProcesado::ErrorProcesando{
			    ruta_pagina: ruta_pagina.clone(),
			    numero_pagina: numero_pagina,
			    motivo: motivos.join(",")
			}
		    );


		}
		
	    }

	    Ok(cola_documentos_a_generar)

	})

	
    }

    fn procesar_LiquidacionAnticipos(&mut self, sesion: &Sesion, db: &std::sync::Arc<sled::Db>, estado_procesos_mutex: &std::sync::Arc<std::sync::Mutex<std::collections::HashMap<String,String>>>) -> Result<(), Box<dyn std::error::Error>>{
	
	Self::act_est_proc(self.id(),estado_procesos_mutex,format!("Separando PDF en paginas..."))?;

	let id = self.id();
	
	self.procesar_pdf_multipagina(sesion, db, TipoDocProcesado::LiquidacionAnticipos, estado_procesos_mutex, |mut rutas_paginas| {

	    let total_paginas = rutas_paginas.len();
	    
	    //Compilar las expresiones regulares para encontrlet re_nit = regex::Regex::new(r#"Nit \:[ ]{4,}(\d{8,})"#).expect("La expresion regular de nit liquidacion de cana esta mala");ar en cada pagina
	    //los campos que identifican cada liquidacion
	    let re_fecha = regex::Regex::new(r#"FECHA CONTABLE[ ]{4,}(\d{2})/(\d{2})/(\d{4})"#).expect("La expresion regular de fechas liquidacion de cana esta mala");
		let re_fecha_aux = regex::Regex::new(r#"FECHA CONTABLE[ ]{4,}(\d{2})/(\d{1})/(\d{4})"#).expect("La expresion regular de fechas liquidacion de cana esta mala");
		let re_nit = regex::Regex::new(r#"Nit[ ]{4,}(\d{7,})"#).expect("La expresion regular de nit liquidacion de cana esta mala");
	    let re_hacienda = regex::Regex::new(r#"Hacienda[ ]{4,}(\d{3,})"#).expect("La expresion regular de hacienda liquidacion de cana esta mala");
	    let re_cuenta_proveedor = regex::Regex::new(r#"Cuenta SAP[ ]{4,}(\d{5,})"#).expect("La expresion regular de cuenta proveedor liquidacion de cana esta mala");
	    let mut cola_documentos_a_generar : Vec<ItemProcesado> = vec![];

	    
	    //Iterar por todas pdf las paginas
	    for (i, ruta_pagina) in rutas_paginas.iter().enumerate() {

		Self::act_est_proc(id.clone(),estado_procesos_mutex,format!("Procesando campos pagina {} de {}",i,total_paginas))?;

		//Obtener el texto del pdf de la pagina
		let contenido_pagina = crate::utils::convertir_pdf_en_string( ruta_pagina.as_path() )?;

		//Encontrar la fecha
		//FECHA CONTABLE                   31/01/2021
		let mut fecha : Vec<String> = Default::default();
		for cap in re_fecha.captures_iter(&contenido_pagina) {
		    fecha.push(format!("{}-{}-{}",cap[3].to_string() ,cap[2].to_string(),cap[1].to_string()));
		}

		//Encontrar la fecha  si el codigo de linea 805 falla este es para los mes que no tiene 2 digitos
		//      30/1/2020
		if fecha.len() != 1 {
			for cap in re_fecha_aux.captures_iter(&contenido_pagina) {

				let fech_cap: String = cap[2].to_string();
    			let adj: String = "0".to_string();
				let fec_  =  format!("{}{}", adj, fech_cap);

				fecha.push(format!("{}-{}-{}",cap[3].to_string() , fec_ ,cap[1].to_string()));
			}
		}

		//Encontrar NIT
		//Nit           8903009516
		let mut nit : Vec<String> = Default::default();
		for cap in re_nit.captures_iter(&contenido_pagina) {
		    nit.push(cap[1].to_string());
		}

		//Encontrar Hacienda
		//Hacienda      050 ESPERANZA MUNCHMEYER
		let mut hacienda : Vec<String> = Default::default();
		for cap in re_hacienda.captures_iter(&contenido_pagina) {
		    hacienda.push(cap[1].to_string());
		}

		//Encontrar cuenta proveedor
		//Cuenta SAP    2000810
		let mut cuenta_proveedor : Vec<String> = Default::default();
		for cap in re_cuenta_proveedor.captures_iter(&contenido_pagina) {
		    cuenta_proveedor.push(cap[1].to_string());
		}

		//Si encuentra solo una instancia de todos los campos
		//agregar al listado a generar; en caso contrario fallar
		//indicando el numero de pagina y el campo
		let numero_pagina = if let Some(stem) = ruta_pagina.file_stem() { stem.to_string_lossy().to_string().split("_").nth(1).unwrap_or("(numero pagina no se pudo determinar)").to_string() } else { "(numero pagina no se pudo determinar)".to_string() };
		if fecha.len() == 1 && nit.len() == 1 && hacienda.len() == 1 && cuenta_proveedor.len() == 1{

		    cola_documentos_a_generar.push(
			ItemProcesado::CorrectamenteProcesado{
			    ruta_pagina: ruta_pagina.clone(),
			    llaves: vec![
				nit[0].to_string(),
				fecha[0].to_string(),
				hacienda[0].to_string(),
				cuenta_proveedor[0].to_string()
			    ],
			    numero_pagina: numero_pagina,
			}
		    );

		    
		}else{
		    
		    let mut motivos :Vec<&str> = vec![];

		    if fecha.len() != 1 { motivos.push("fecha no encontrada"); }
		    if nit.len() != 1 { motivos.push("nit no encontrado"); }
		    if cuenta_proveedor.len() != 1 { motivos.push("cuenta proveedor no encontrada"); }
		    if hacienda.len() != 1 { motivos.push("hacienda no encontrada"); }
		    
		    cola_documentos_a_generar.push(
			ItemProcesado::ErrorProcesando{
			    ruta_pagina: ruta_pagina.clone(),
			    numero_pagina: numero_pagina,
			    motivo: motivos.join(",")
			}
		    );

		}
		
	    }

	    Ok(cola_documentos_a_generar)

	})

	
    }

    
	fn procesar_LiquidacionCanaParticipacion(&mut self, sesion: &Sesion, db: &std::sync::Arc<sled::Db>, estado_procesos_mutex: &std::sync::Arc<std::sync::Mutex<std::collections::HashMap<String,String>>>) -> Result<(), Box<dyn std::error::Error>>{
	
		Self::act_est_proc(self.id(),estado_procesos_mutex,format!("Separando PDF en paginas..."))?;
	
		let id = self.id();
		
		self.procesar_pdf_multipagina(sesion, db, TipoDocProcesado::LiquidacionCanaParticipacion, estado_procesos_mutex, |mut rutas_paginas| {
	
			let total_paginas = rutas_paginas.len();
			
			//Compilar las expresiones regulares para encontrlet re_nit = regex::Regex::new(r#"Nit \:[ ]{4,}(\d{8,})"#).expect("La expresion regular de nit liquidacion de cana esta mala");ar en cada pagina
			//los campos que identifican cada liquidacion
			let re_fecha = regex::Regex::new(r#"(\d{2})/(\d{2})/(\d{4})"#).expect("La expresion regular de fechas liquidacion de cana esta mala");
			let re_fecha_aux = regex::Regex::new(r#"(\d{2})/(\d{1})/(\d{4})"#).expect("La expresion regular de fechas liquidacion de cana esta mala");
			let re_nit = regex::Regex::new(r#"Nit :[ ]{4,}(\d{7,})"#).expect("La expresion regular de nit liquidacion de cana esta mala");
			let re_hacienda = regex::Regex::new(r#"Hacienda :[ ]{4,}(\d{3,})"#).expect("La expresion regular de hacienda liquidacion de cana esta mala");
			let re_cuenta_proveedor = regex::Regex::new(r#"Cuenta Participe :[ ]{4,}(\d{3,})"#).expect("La expresion regular de cuenta proveedor liquidacion de cana esta mala");
			let mut cola_documentos_a_generar : Vec<ItemProcesado> = vec![];
	
			
			//Iterar por todas pdf las paginas
			for (i, ruta_pagina) in rutas_paginas.iter().enumerate() {
	
			Self::act_est_proc(id.clone(),estado_procesos_mutex,format!("Procesando campos pagina {} de {}",i,total_paginas))?;
	
			//Obtener el texto del pdf de la pagina
			let contenido_pagina = crate::utils::convertir_pdf_en_string( ruta_pagina.as_path() )?;
		   //  std::fs::write("/home/eramirez/portal_proveedores_carmelita_backend/salida.txt",&contenido_pagina); 
			//Encontrar la fecha
			//      30/11/2020
			let mut fecha : Vec<String> = Default::default();
			for cap in re_fecha.captures_iter(&contenido_pagina) {
				fecha.push(format!("{}-{}-{}",cap[3].to_string() ,cap[2].to_string(),cap[1].to_string()));
			}
			//Encontrar la fecha  si el codigo de linea 578 falla este es para los mes que no tiene 2 digitos
			//      30/1/2020
			if fecha.len() != 1 {
				for cap in re_fecha_aux.captures_iter(&contenido_pagina) {
	
					let fech_cap: String = cap[2].to_string();
					let adj: String = "0".to_string();
					let fec_  =  format!("{}{}", adj, fech_cap);
	
					fecha.push(format!("{}-{}-{}",cap[3].to_string() , fec_ ,cap[1].to_string()));
				}
			}
	
			//Encontrar NIT
			//Nit :                                               8913002331
			let mut nit : Vec<String> = Default::default();
			for cap in re_nit.captures_iter(&contenido_pagina) {
				nit.push(cap[1].to_string());
			}
	
			//Encontrar Hacienda
			//Hacienda :              300
			let mut hacienda : Vec<String> = Default::default();
			for cap in re_hacienda.captures_iter(&contenido_pagina) {
				hacienda.push(cap[1].to_string());
			}
	
			//Encontrar cuenta proveedor
			//Cuenta Proveedor :                                  2000800
			let mut cuenta_proveedor : Vec<String> = Default::default();
			for cap in re_cuenta_proveedor.captures_iter(&contenido_pagina) {
				cuenta_proveedor.push(cap[1].to_string());
			}
	
			//Si encuentra solo una instancia de todos los campos
			//agregar al listado a generar; en caso contrario fallar
			//indicando el numero de pagina y el campo
			let numero_pagina = if let Some(stem) = ruta_pagina.file_stem() {
				stem
				.to_string_lossy()
				.to_string()
				.split("_")
				.nth(1)
				.unwrap_or("(numero pagina no se pudo determinar)")
				.to_string()
			} else {
				"(numero pagina no se pudo determinar)".to_string()
			};
			
			if fecha.len() == 1 && nit.len() == 1 && hacienda.len() == 1 && cuenta_proveedor.len() == 1{
	
				cola_documentos_a_generar.push(
				ItemProcesado::CorrectamenteProcesado{
					ruta_pagina: ruta_pagina.clone(),
					llaves: vec![
					nit[0].to_string(),
					fecha[0].to_string(),
					hacienda[0].to_string(),
					cuenta_proveedor[0].to_string()
					],
					numero_pagina: numero_pagina,
				}
				);
				
			}else{
	
				let mut motivos :Vec<&str> = vec![];
	
				if fecha.len() != 1 { motivos.push("fecha no encontrada"); }
				if nit.len() != 1 { motivos.push("nit no encontrado"); }
				if cuenta_proveedor.len() != 1 { motivos.push("cuenta proveedor no encontrada"); }
				
				cola_documentos_a_generar.push(
				ItemProcesado::ErrorProcesando{
					ruta_pagina: ruta_pagina.clone(),
					numero_pagina: numero_pagina,
					motivo: motivos.join(",")
				}
				);
			}
			
			}
	
			Ok(cola_documentos_a_generar)
	
		})
	
		}
    

    fn procesar_Ninguno(&mut self) -> Result<(), Box<dyn std::error::Error>>{
	Ok(())
    }


    
}
