use serde_derive::{Deserialize, Serialize};
use super::{Auditable, Storable};
use crate::{
    errors,
    utils::ConfiguracionServidorFirebase,
    model::{
	usuario::Usuario
    }
};

#[derive(Serialize, Clone, Debug)]
pub enum TipoNotificacionPush{
    DonacionesCenicana,
    FondoSocial,
    LiquidacionCana,
    LiquidacionMercadoExcedentario,
    LiquidacionAnticipos,
    Retenciones,
    IngresosYCostos,
    Ica,
    Ninguno,
    EntradaDeCana,
    InformeDeProduccion,
	LiquidacionCanaParticipacion,
}

pub struct NotificacionPush{
    titulo: String,
    mensaje: String,
    tipo: TipoNotificacionPush,
    token_destino: String,
    id: String,
}

#[derive(Serialize, Debug, Clone)]
struct FirebaseNotification{
    title: String,
    body: String,
}

#[derive(Serialize, Debug, Clone)]
struct FirebasePushMessageData{
    click_action:String,
    tipo: TipoNotificacionPush,
    body: String,
}



#[derive(Serialize, Debug, Clone)]
struct FirebasePushMessage{
    to:String,
    notification: FirebaseNotification,
    data: FirebasePushMessageData,
}

#[derive(Deserialize, Clone, Debug)]
struct FirebasePushMessageResponse{
    multicast_id: u64,
    success: u32,
    failure: u32,
    canonical_ids: u32,
    results: serde_json::Value,
}


#[derive(Serialize,Deserialize,Debug)]
pub struct ConsultaNotificacionesSICAP{
    API_KEY:String,
}

#[derive(Serialize, Debug)]
pub struct BorrarNotificacionesSICAP{
    IDS: Vec<String>,
    API_KEY:String,
}


#[derive(Serialize,Deserialize,Debug)]
pub struct FilaConsultaNotificacionesEntradaCanaSICAP{
    FAZ: String,
    LOTE: String,
    TAL: String,
    SUERTE: String,
    GUIA: String,
    EQUIPO: String,
    CANASTA: String,
    FCH_ENT: String,
    HR_ENT: String,
    TC: String,
    ID: String,
    NIT: String,
}


#[derive(Serialize,Deserialize,Debug)]
pub struct FilaConsultaNotificacionesProduccionSICAP{
    FAZ: String,
    LOTE: String,
    TAL: String,
    SUERTE: String,
    TC: String,
    TCH: String,
    TCHM: String,
    FCH_CORTE: String,
    EDAD_COS: String,
    DATOS: String,
    CPF: String,
    ID: String,
    NIT: String,
}

#[derive(Serialize,Deserialize,Debug)]
pub enum FilaConsultaNotificacionSICAP{
    EntradaCana(FilaConsultaNotificacionesEntradaCanaSICAP),
    Produccion(FilaConsultaNotificacionesProduccionSICAP)
}


impl NotificacionPush{
    
    pub fn new(titulo: String,mensaje: String, tipo: TipoNotificacionPush, token_destino: String, id: String) -> Self {
	Self{
	    titulo,
	    mensaje,
	    tipo,
	    token_destino,
	    id,
	}
    }

    fn convertir_a_firebase(&self) -> FirebasePushMessage{
	
	FirebasePushMessage{
	    to:self.token_destino.to_string(),
	    notification: FirebaseNotification{
		title: self.titulo.to_string(),
		body: self.mensaje.to_string(),
	    },
	    data: FirebasePushMessageData{
		click_action: "FLUTTER_NOTIFICATION_CLICK".to_string(),
		tipo: self.tipo.clone(),
		body: self.mensaje.to_string(),
	    },
	}
    }

    pub fn enviar(&self, config:&ConfiguracionServidorFirebase ) -> Result<(), errors::ErroresNotificacionesPush> {

	println!("Enviando notificacion push {:?}",serde_json::to_string(&self.convertir_a_firebase()) );
	
	let response = reqwest::blocking::Client::new()
	    .post(&format!("{url}",url=config.url))
	    .header("Authorization",config.api_key.clone())
	    .json(&dbg!(self.convertir_a_firebase()))
	    .send()
	    .map_err(|e| errors::ErroresNotificacionesPush::ErrorCreandoSolicitud{motivo: e.to_string()} )?;

	let response_text  = response.text()
	    .map_err(|e|{			
		errors::ErroresNotificacionesPush::ErrorLeyendoResultado{motivo: e.to_string(), contenido_respuesta: "".to_string()}
	    })?;
	
	let respuesta_firebase : FirebasePushMessageResponse = serde_json::from_str(&response_text)
	    .map_err(|e|{			
		errors::ErroresNotificacionesPush::ErrorLeyendoResultado{motivo: e.to_string(), contenido_respuesta: response_text}
	    })?;
	
	if respuesta_firebase.failure > 0 {
	    
	    Err( errors::ErroresNotificacionesPush::ErrorEnvioMensajeAFirebase{motivo: respuesta_firebase.results.to_string() } )
		
	}else{
	    
	    Ok( () )
		
	}

    }

    pub fn obtener_tokens_por_nit(db: &std::sync::Arc<sled::Db>, nit: &str) -> Vec<String>{
	Usuario::scan_map(&db,|(i,x)|{
	    if x.nits().iter().any(|y| y == nit) {
		Some(x.token_firebase())
	    } else {
		None
	    }
	})
    }

    pub fn iniciar_motor_notificaciones(config: std::sync::Arc<crate::utils::ArchivoConfiguracion>, db: std::sync::Arc<sled::Db>) -> Result<(), errors::ErroresConfiguracion >{

	let cron_espera_consulta_sicap_notificaciones_push : job_scheduler::Schedule = config.procesos_fondo.cron_espera_consulta_sicap_notificaciones_push
	    .parse::<job_scheduler::Schedule>()
	    .map_err(|e| errors::ErroresConfiguracion::FallaInterpretacionCron{cron:"cron_espera_consulta_sicap_notificaciones_push".to_string(), motivo: e.to_string() })?;


	let _ = std::thread::spawn(move ||{

	    let mut sched = job_scheduler::JobScheduler::new();

	    sched.add(job_scheduler::Job::new( cron_espera_consulta_sicap_notificaciones_push,||{

		//println!("Ejecutando Proceso de fondo consulta notificaciones SICAP...");
		
		let consulta = ConsultaNotificacionesSICAP{
		    API_KEY:config.api_key.to_string(),
		};
		
		let cliente_lectura = reqwest::blocking::Client::new();
		
		match cliente_lectura.post(&format!("{url}/notificaciones",url=config.url_proxy))
		    .json(&consulta)
		    .send(){
			Ok(response) =>{

			    match response.json::<Vec<FilaConsultaNotificacionSICAP>>(){
				Ok(respuesta_sicap) => {

				    let mut a_borrar : Vec<String> = vec![];
				    
				    for fila in respuesta_sicap.iter(){

					match fila {
					    FilaConsultaNotificacionSICAP::EntradaCana(fila_interna)=>{

						let tokens_destino : Vec<String> = Self::obtener_tokens_por_nit(&db, &fila_interna.NIT);

						for token in tokens_destino.iter(){

						    let mut notificacion = Self::new(
							format!(
							    "Entrada de Caña {} {}",
							    fila_interna.FCH_ENT.to_string(),
							    fila_interna.HR_ENT.to_string(),
							),
							format!(
							    "Hacienda {}, Suerte {}, Placa {}, Canasta {}, Peso {}",
							    fila_interna.FAZ.to_string(),
							    fila_interna.SUERTE.to_string(),
							    fila_interna.EQUIPO.to_string(),
							    fila_interna.CANASTA.to_string(),
							    fila_interna.TC.to_string(),
							),
							TipoNotificacionPush::EntradaDeCana,
							token.to_string(),
							fila_interna.ID.to_string(),
						    );

						    if let Err(why) = notificacion.enviar(&config.firebase){
							println!("Falla al enviar notificacion entrada de caña a firebase: {}", why.to_string());
						    }else{
							a_borrar.push(fila_interna.ID.to_string());
						    }
						}

					    },
					    FilaConsultaNotificacionSICAP::Produccion(fila_interna)=>{

						let tokens_destino : Vec<String> = Self::obtener_tokens_por_nit(&db, &fila_interna.NIT);

						for token in tokens_destino.iter(){

						    
						    let mut notificacion = Self::new(
							format!(
							    "Resultados de producción disponibles"
							),
							format!(
							    "Ya se encuentra disponible para consultar en la aplicación, los resultados de producción para la Hacienda {}, Suerte {}.",
							    fila_interna.FAZ.to_string(),
							    fila_interna.SUERTE.to_string()
							),
							TipoNotificacionPush::InformeDeProduccion,
							token.to_string(),
							fila_interna.ID.to_string(),
						    );
						    
						    if let Err(why) = notificacion.enviar(&config.firebase){
							println!("Falla al enviar notificacion informe de produccion a firebase: {}", why.to_string());
						    }else{
							a_borrar.push(fila_interna.ID.to_string());
						    }
						}
						    
						    
					    },
					}
				    }

				    //Mandar a borrar todas la notificaciones enviadas
				    a_borrar.dedup();
				    let notificaciones_borrar = BorrarNotificacionesSICAP{
					IDS: a_borrar.clone(),
					API_KEY:config.api_key.to_string(),
				    };

				    
				    let cliente_borrado = reqwest::blocking::Client::new();
				    match cliente_lectura.post(&format!("{url}/borrar_notificaciones",url=config.url_proxy))
					.json(&dbg!(notificaciones_borrar))
					.send(){
					    Ok(response) =>{
						//println!("Borrado de notifiaciones efectuado sin errores");
					    },
					    Err(why) => {
						println!("Error el solicitar borrar notifiaciones: {}", why.to_string())
					    }
					}
				    

				},
				Err(why) =>{
				    println!("Falla deserializando respuesta SICAP:{:?}", why.to_string());
				}
			    };
			    
			},
			Err(why) => {
			    println!("Falla intentando realizar peticion HTTP a SICAP:{:?}", why.to_string());
			}
		    }
		
	    }));

	    
	    loop{
		sched.tick();
		std::thread::sleep(std::time::Duration::from_millis(config.procesos_fondo.tiempo_espera_ticks_motor_milisegundos));
	    }
	    
	});

	Ok( () )


    }
    
    
}


