use chrono::prelude::*;
use serde_derive::{Deserialize, Serialize};
use crate::errors;
use crate::utils;

pub mod usuario;
pub mod sesion;
pub mod objetos_seguridad;
pub mod refresh_token;
pub mod rol;
pub mod documento;
pub mod orden_procesamiento_documentos;
pub mod asistencia_tecnica;
pub mod notificaciones_push;
pub mod trazabilidad_plataforma;
pub mod log_politica;
pub mod pqrs;

#[macro_export]
macro_rules! audit{
    ($struct_name:ty) => {

	impl crate::model::Auditable for $struct_name {
	    fn creation_stamp(&self) -> i64 {
		self.creation_stamp
	    }
	    
	    fn modification_stamp(&self) -> i64{
		self.modification_stamp
	    }
	    fn created_by(&self) -> String{
		self.created_by.to_string()
	    }
	    
	    fn modified_by(&self) -> String{
		self.modified_by.to_string()
	    }
	    
	    fn set_creation_stamp(&mut self, when:i64){
		self.creation_stamp = when;
	    }
	    
	    fn set_modification_stamp(&mut self, when:i64){
		self.modification_stamp = when;
	    }
	    fn set_created_by(&mut self, sesion:&crate::model::sesion::Sesion){
		self.created_by = sesion.usuario().id();
	    }
	    fn set_modified_by(&mut self, sesion:&crate::model::sesion::Sesion){
		self.modified_by = sesion.usuario().id();
	    }

	}
    }
}

#[macro_export]
macro_rules! store {
    ($struct_name:ty, $id_field:ident, $key_list:expr, $index_list:expr ) => {

	
	impl crate::model::Storable for $struct_name {

	    fn index_list() -> Vec<&'static str>{
		$index_list
	    }

	    fn keys() -> Vec<&'static str>{
		$key_list
	    }

	    fn id_prefix() -> String {
		stringify!($struct_name).to_string()
	    }
	    
	    fn id(&self) -> String{
		self.$id_field.to_string()
	    }

	    fn set_id(&mut self, id:&str){
		self.$id_field = id.to_string();
	    }

	    fn to_raw(&self) -> Vec<u8>{
		rmp_serde::to_vec(self).unwrap()
	    }

	    fn from_raw(raw_data: Vec<u8>) -> Result<Self, Box< dyn std::error::Error> > {
		Ok( rmp_serde::from_read_ref(&raw_data).map_err(|e| Box::new(errors::StorableError::ReadError( e.to_string() )) )? )
	    }

	    fn import_tab_stream_reader(db : &sled::Db, s_in: &dyn std::io::Read) -> Result<(), Box<dyn std::error::Error>>{
		Ok(())
	    }

	    fn export_tab_stream_writer(db : &sled::Db, s_out: &dyn std::io::Write) -> Result<(), Box<dyn std::error::Error>>{
		Ok(())
	    }


	}
    }
}

use sesion::Sesion;

pub trait Auditable{
    fn creation_stamp(&self) -> i64;
    fn modification_stamp(&self) -> i64;
    fn created_by(&self) -> String;
    fn modified_by(&self) -> String;
    fn set_creation_stamp(&mut self, when:i64);
    fn set_modification_stamp(&mut self, when:i64);
    fn set_created_by(&mut self, id:&crate::model::sesion::Sesion);
    fn set_modified_by(&mut self, id:&crate::model::sesion::Sesion);

    fn record_modification(&mut self, sesion: &Sesion, cuando: i64){
	self.set_modified_by(sesion);
	self.set_modification_stamp(cuando);
    }
    
    fn record_creation(&mut self, sesion: &Sesion, cuando: i64){
	self.set_created_by(sesion);
	self.set_creation_stamp(cuando);
    }

}

pub trait Storable : Auditable {

    fn id(&self) -> String;

    fn set_id(&mut self, id:&str);

    fn generate_id() -> String {
	Self::id_prefix() + &utils::generate_id()
    }
    
    fn create_id(&mut self) {
	let id = Self::generate_id();
	self.set_id(&id);
    }

    fn check_before_store(&self, _db : &sled::Db) -> Result<(), Box<dyn std::error::Error>>{
	Ok(())
    }

    fn id_prefix() -> String;

    fn keys() -> Vec<&'static str>;

    fn index_list() -> Vec<&'static str>;

    fn to_raw(&self) -> Vec<u8>;

    fn from_raw(raw_data: Vec<u8>) -> Result<Self, Box<dyn std::error::Error>> where Self: Sized;

    fn import_tab_stream_reader(db : &sled::Db, s_in: &dyn std::io::Read) -> Result<(), Box<dyn std::error::Error>> where Self: Sized;

    fn export_tab_stream_writer(db : &sled::Db, s_out: &dyn std::io::Write) -> Result<(), Box<dyn std::error::Error>> where Self: Sized;

    fn check_exists(db : &sled::Db, id:&str) -> bool {
	if let Ok(exists) = db.contains_key(id){
	    
	    exists

	} else {
	    
	    false
		
	}
	
    }
    
    fn load( db : &sled::Db, id:&str ) -> Result<Option<Self>, Box<dyn std::error::Error>> where Self: Sized {
	
	if let Some(raw_data) = db.get(id).map_err(|e| errors::StorableError::ReadError( e.to_string() ) )? {
	    
	    Ok( Some( Self::from_raw( raw_data.to_vec() )? ) )

	} else {

	    Ok( None )
	}
				       
    }

    fn scan<T>(db : &sled::Db, conditional: T ) -> Vec<Self>
    where
	Self: Sized,
        T : Fn(&Self) -> bool
    {
	let mut values_to_return = vec![];
	
	for raw_item in db.scan_prefix(Self::id_prefix()){
	    
	    if let Ok((_key,raw)) = raw_item {

		if let Ok(item) = Self::from_raw( raw.to_vec() ) {

		    if conditional(&item) == true {
			values_to_return.push(item);
		    }
		    
		}
	    }
	}

	return values_to_return;
    }

    fn scan_first<T>(db : &sled::Db, conditional: T ) -> Option<Self>
    where
	Self: Sized,
        T : Fn(&Self) -> bool
    {
	
	for raw_item in db.scan_prefix(Self::id_prefix()){
	    
	    if let Ok((_key,raw)) = raw_item {

		if let Ok(item) = Self::from_raw( raw.to_vec() ) {

		    if conditional(&item) == true {
			return Some(item);
		    }
		    
		}
	    }
	}

	return None;
    }

    fn scan_map<T,U>(db : &sled::Db, mut conditional: T ) -> Vec<U>
    where
	Self: Sized,
        U : Clone,
        T : FnMut((usize, &Self)) -> Option<U>
    {
	let mut values_to_return = vec![];
	
	for raw_item in db.scan_prefix(Self::id_prefix()).enumerate(){
	    
	    if let (i,Ok((_key,raw))) = raw_item {

		if let Ok(item) = Self::from_raw( raw.to_vec() ) {

		    if let Some(val) = conditional((i,&item)) {
			values_to_return.push(val.clone());
		    }
		    
		}
	    }
	}

	return values_to_return;
    }

	
    fn store(&mut self, db : &sled::Db, sesion: &Sesion) -> Result<(), Box< dyn std::error::Error> > {

	self.check_before_store(db)?;
	
	let now = crate::utils::now_microseconds();

	let mut is_update = false;

	if let Some(_p) = db.get(self.id())? {

	    self.record_modification(sesion, now);
	    is_update = true;
	    
	} else {

	    self.create_id();
	    self.record_creation(sesion, now);

	}

	let raw = self.to_raw();
	db.insert( self.id(), raw)?;

	if is_update == false{
	    self.store_index(db)?;
	}
	
	Ok(())
    }

    fn store_index(&mut self, db : &sled::Db) -> Result<(), Box< dyn std::error::Error> >{
	Ok(())
    }
    
    fn delete(&self, db : &sled::Db) -> Result<(), Box< dyn std::error::Error> >{
	db
	    .remove( self.id() )
	    .map_err(|e| Box::new(errors::StorableError::WriteError( e.to_string() )))?;
	
	Ok(())
    }

    fn delete_all(db : &sled::Db) -> Result<(), Box< dyn std::error::Error> > {

	let mut keys_to_delete = vec![];
	
	for item in db.scan_prefix(Self::id_prefix()){
	    
	    if let Ok((key,_)) = item {
		
		keys_to_delete.push(key);
		
	    }
	}

	for item in keys_to_delete {
	    
	    db.remove(item).map_err(|e| Box::new(errors::StorableError::WriteError( e.to_string() )))?;
	    
	}

	Ok(())

	
    }

}



#[derive(Deserialize, Serialize, Clone, Debug)]
pub enum Ubicacion{
    Recepcion,
    Bodega,
    Trilla,
    Distribuidor,
}


