use chrono::prelude::*;
use serde_derive::{Deserialize, Serialize};
use crate::errors;
use super::usuario::Usuario;
use super::{Storable};
use crate::{audit, store};
use std::default::Default;
audit!{Documento}

const llave_busqueda : &str = "BUSDOC";

#[derive(Deserialize, Serialize, Debug, Clone, PartialEq)]
pub enum TipoDocProcesado{
    DonacionesCenicana,
    FondoSocial,
    LiquidacionCana,
    LiquidacionMercadoExcedentario,
    LiquidacionAnticipos,
    Retenciones,
    IngresosYCostos,
    Ica,
    Ninguno,
    LiquidacionCanaParticipacion,
}

impl Default for TipoDocProcesado{
    fn default() -> Self{
	Self::Ninguno
    }
}

impl std::fmt::Display for TipoDocProcesado {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
	let text = match self {
	    TipoDocProcesado::DonacionesCenicana => "Donaciones Cenicaña",
	    TipoDocProcesado::FondoSocial => "Donaciones Fondo Social",
	    TipoDocProcesado::LiquidacionCana => "Liquidacion Caña",
	    TipoDocProcesado::LiquidacionMercadoExcedentario => "Liquidacion Mercado Excedentario",
	    TipoDocProcesado::LiquidacionAnticipos => "Liquidacion Anticipos",
	    TipoDocProcesado::Retenciones => "Retenciones",
	    TipoDocProcesado::IngresosYCostos => "Ingresos Y Costos",
	    TipoDocProcesado::Ica => "ICA",
	    TipoDocProcesado::Ninguno => "Ninguno",
        TipoDocProcesado::LiquidacionCanaParticipacion => "Liquidacion Participacion",
	};
	
        write!(f, "{}", text)
    }
}


#[derive(Deserialize, Serialize, Debug, Clone)]
pub struct Documento{
    id: String,
    llaves: Vec<String>,
    tipo: TipoDocProcesado,
    fecha_distribucion: Option<i64>,
    fecha_documento:  Option<i64>,
    orden_procesamiento_documentos_id:String,
    creation_stamp: i64,
    created_by: String,
    modification_stamp: i64,
    modified_by: String,
}

impl Default for Documento {
    fn default () -> Self {
	Self{
	    id: Default::default(),
	    tipo: Default::default(),
	    llaves: Default::default(),
	    fecha_distribucion: None,
	    fecha_documento: None,
	    orden_procesamiento_documentos_id: Default::default(),
	    creation_stamp: Utc::now().timestamp(),
	    created_by: Default::default(),
	    modification_stamp: Utc::now().timestamp(),
	    modified_by: Default::default(),
	}
    }
}

impl crate::model::Storable for Documento {

    fn index_list() -> Vec<&'static str>{
	vec![]
    }

    fn keys() -> Vec<&'static str>{
	vec![]
    }

    fn id_prefix() -> String {
	stringify!(Documento).to_string()
    }
    
    fn id(&self) -> String{
	self.id.to_string()
    }

    fn set_id(&mut self, id:&str){
	self.id = id.to_string();
    }

    fn to_raw(&self) -> Vec<u8>{
	rmp_serde::to_vec(self).unwrap()
    }

    fn from_raw(raw_data: Vec<u8>) -> Result<Self, Box< dyn std::error::Error> > {
	Ok( rmp_serde::from_read_ref(&raw_data).map_err(|e| Box::new(errors::StorableError::ReadError( e.to_string() )) )? )
    }

    fn import_tab_stream_reader(db : &sled::Db, s_in: &dyn std::io::Read) -> Result<(), Box<dyn std::error::Error>>{
	Ok(())
    }

    fn export_tab_stream_writer(db : &sled::Db, s_out: &dyn std::io::Write) -> Result<(), Box<dyn std::error::Error>>{
	Ok(())
    }

    fn store_index(&mut self, db : &sled::Db) -> Result<(), Box<dyn std::error::Error>>{
	let llave = llave_busqueda.to_string() + &self.llaves().join("");
	let contenido = self.id();
	
	db.insert(&llave[..], &contenido[..] )?;

	Ok(())

    }


}


impl Documento {
    pub fn new() -> Self {
	Default::default()
    }

    pub fn llave(&self) -> String{
	self.llaves.join("")
    }

    pub fn llaves(&self) -> Vec<String>{
	self.llaves.clone()
    }
    
    pub fn tipo(&self) -> TipoDocProcesado {
	self.tipo.clone()
    }
    
    pub fn fecha_distribucion(&self) -> Option<i64> {
	self.fecha_distribucion
    }

    pub fn fecha_documento(&self) -> Option<i64> {
	self.fecha_documento
    }

    pub fn orden_procesamiento_documentos_id(&self) -> String{
	self.orden_procesamiento_documentos_id.to_string()
    }
    
    pub fn data(&self, db: &std::sync::Arc<sled::Db>) -> Result<sled::IVec, Box<std::error::Error>> {
	crate::utils::leer_blob(db, &self.id)
    }
        
    pub fn set_tipo(&mut self, i:TipoDocProcesado){
	self.tipo = i;
    }

    pub fn set_llaves(&mut self, i:Vec<String>){
	self.llaves = i;
	
    }

    
    pub fn set_fecha_distribucion(&mut self, i:Option<i64>){
	self.fecha_distribucion = i;
    }

    pub fn set_fecha_documento(&mut self, i:Option<i64>){
	self.fecha_documento = i;
    }

    
    pub fn set_orden_procesamiento_documentos_id(&mut self, i:&str){
	self.orden_procesamiento_documentos_id = i.to_string();
    }
    
    pub fn set_data(&mut self, db: &std::sync::Arc<sled::Db>,i: Vec<u8>) -> Result<(), Box<dyn std::error::Error>>{
	crate::utils::guardar_blob(db, &self.id, i)
    }

    pub fn delete_data(&mut self, db: &std::sync::Arc<sled::Db>) -> Result<(), Box<dyn std::error::Error>>{
	crate::utils::borrar_blob(db, &self.id)
    }

    pub fn buscar_por_llaves(llaves: Vec<String>, db : &sled::Db) -> Result<Self, Box<dyn std::error::Error>> {

	let llave = llave_busqueda.to_string() + &llaves.clone().join("");
	
	let raw_id = db.get(&llave[..])?
	    .ok_or(Box::new(errors::ProcesamientoDocumentos::DocumentoNoEncontrado{llaves: llaves.join("") }))?;

	let id = std::str::from_utf8(&raw_id)?.to_string();

	Self::load(db, &id)?
	    .ok_or(Box::new(errors::ProcesamientoDocumentos::DocumentoNoEncontrado{llaves: id }))
    }

}
