#![feature(backtrace)]
pub mod handlers;
pub mod tls_certificates;
pub mod model;
pub mod routes;
pub mod interfaces;
pub mod errors;
pub mod utils;

use std::sync::{atomic::{AtomicUsize, Ordering}};

use model::{
    usuario::Usuario,
    Storable,
    objetos_seguridad::ObjetosSeguridad,
    rol::Rol,
};
use utils::{ArchivoConfiguracion, ConfiguracionBackendSICAP};
use job_scheduler::{JobScheduler, Job};
use std::time::Duration;

#[tokio::main]
async fn main(){

    //https://redislabs.com/blog/get-sql-like-experience-redis/

    pretty_env_logger::init();

    const VERSION: &'static str = env!("CARGO_PKG_VERSION");
    const AUTHORS: &'static str = env!("CARGO_PKG_AUTHORS");
    const DESCRIPTION: &'static str = env!("CARGO_PKG_DESCRIPTION");

    let app = clap::App::new("Portal Proveedores Caña Carmelita: Servidor de Aplicacion Web")
        .version(VERSION)
        .author(AUTHORS)
        .about(DESCRIPTION)
        .arg(clap::Arg::with_name("plain-mode")
             .short("p")
             .long("plain-mode")
             .help("Inicir el servidor en modo HTTP simple para generar certificado TLS. Este parametro solo se puede usar si no existe archivo de configuracion. (Los certificados son generados en el directorio ./tls )"))
        .arg(clap::Arg::with_name("tls-mode")
	     .short("t")
	     .long("tls-mode")
             .help("Inicia el servidor con encrypcion TLS. Este parametro solo se puede usar si no existe archivo de configuracion. (Los certificados son ledidos del directorio ./tls)"))
        .arg(clap::Arg::with_name("port")
             .short("b")
             .long("port")
             .help("Establece el puerto TCP en donde sel servidor HTTP escucha Este parametro solo se puede usar si no existe archivo de configuracion. (Por predefinido 60000)")
	     .takes_value(true)
	     .value_name("PORT_NUMBER"))
	.arg(clap::Arg::with_name("tls-cert-name")
	     .short("c")
             .long("tls-cert-name")
             .help(r#"Nombre del certificado TLS Este parametro solo se puede usar si no existe archivo de configuracion. (Por predefinido "proveedorescarmelita.simpleagri.com")"#)
	     .takes_value(true)
	     .value_name("CERTIFICATE_NAME"))
	.arg(clap::Arg::with_name("tls-cert-account")
             .short("a")
             .long("tls-cert-account")
             .help(r#"Nombre de la cuenta de email ante letsencrypt para generar certificado TLS Este parametro solo se puede usar si no existe archivo de configuracion. (By default "info@carmelita.com.co")"#)
	     .takes_value(true)
	     .value_name("ACCOUNT_NAME"))
	.arg(clap::Arg::with_name("create-user")
             .short("u")
             .long("create-user")
	     .value_name("USER_DATA")
             .help(r#"Crear una nueva cuenta de usuario. Los datos requeridos seguidos de este comando en orden son: <NOMBRE_USUARIO> <CLAVE_ACCESO> "<NOMBRE_COMPLETO>""#)
	     .takes_value(true)
	     .min_values(3))
	.arg(clap::Arg::with_name("create-example-config")
             .short("e")
             .long("create-example-config")
             .help(r#"Crear un archivo de configuracion de ejemplo con el nombre "configuracion.toml". Este parametro solo se puede usar si no existe archivo de configuracion."#));

    
    let mut app2 = app.clone();
    let matches = app.get_matches();

    let tls_mode :bool;
    let mut tls_cert_name = "proveedorescarmelita.simpleagri.com".to_string();
    let mut tls_cert_account = "info@carmelita.com".to_string();
    let mut port : u16 = 60000;
	
    let db = match sled::open("db"){
	Ok(db) => db,
	Err(why) => {
	    println!("\n\nCould not open database file. Reason:{}",why);
	    return;
	}
    };

    /////////////////////////////////////
    // Refresca el token de autenticacion
    // cada 24 horas
    //////////////////////////////////////
    let _ = tokio::spawn(async move {
	    loop{
		println!("refrescando token....");
		crate::refresh_token!(20);
		println!("refresco de token finalizado....");
		std::thread::sleep(std::time::Duration::new(86400,0));
	    }
    });

    
    let db_arc = std::sync::Arc::new(db);


    if let Some(mut user_values) = matches.values_of("create-user"){

	let create_user_failed : bool;
	
	if let Some(user) = user_values.next(){
	    if let Some(password) = user_values.next(){
		if let Some(full_name) = user_values.next(){
		    
		    let mut new_user = match Usuario::crear(
				&db_arc,
				&user,
				&password,
				&full_name,
				"",//telefono1
				"",//telefono2
				"",//telefono3
				"",//email
				"",//email_alternativo
				&vec![],//nits
				&vec![],
            ){
			Ok(x) => x,
			Err(why) => {
			    println!("\n\nFalla al intentar crear usario \"{}\". Motivo {}!", user, why);
			    return;
			}
		    };

		    let sesion = crate::model::sesion::Sesion::new(&new_user);

		    let mut nuevo_rol = Rol::new("AdminSis", "Administrador Absoluto del Sistema");
		    nuevo_rol.agregar_objetos(&mut vec![ObjetosSeguridad::ZZ999]);
		    
		    match nuevo_rol.store(&db_arc, &sesion){
			Ok(_) => (),
			Err(why) => {
			    println!("\n\nFalla al intentar crear rol para usuario \"{}\". Motivo {}!", user, why);
			    return;
			}
		    }

		    new_user.agregar_roles(&mut vec![ nuevo_rol.id() ]);


		    match new_user.store(&db_arc,&sesion){
			Ok(_) => {
			    println!("\n\nUsuario \"{}\" creado exitosamente!", user);
			    return;
			},
			Err(why) => {
			    println!("\n\nFalla al intentar crear usario \"{}\". Motivo {}!", user, why);
			    return;
			}
		    }


		    
		}else{
		    create_user_failed = true;
		}
	    }else{
		
		create_user_failed = true;	
	    }
	}else{
	    create_user_failed = true;
	}

	if create_user_failed == true {
	    app2.print_help().unwrap_or(());
	    println!("\n\nParametros incorrectos para crear nuevo usuario.");
	    return;
	}
    }

    
    //Verificar si el archivo de configuracion existe,
    //en caso tal ejecutar el servidor con los parametros
    //almacenados en este
    let config;
    
    if let Ok(_config_metadata) = std::fs::metadata("configuracion.toml"){

	println!(r#"Archivo de configuracion "configuracion.toml" encontrado! ... intentado interpretar"#);
	match ArchivoConfiguracion::leer_configuracion(){
	    Ok(configuracion) =>{

		println!(r#"Archivo de configuracion "configuracion.toml" interpretado correctamente ... Configurando servidor"#);
		clear_temp(&configuracion);
		config = std::sync::Arc::new(configuracion.clone());
		
		tls_mode = configuracion.tls_mode;
		port = configuracion.puerto_http;
		tls_cert_name = configuracion.tls_cert_name;
		tls_cert_account = configuracion.tls_cert_account;
		
	    },
	    Err(why) => {
		println!(r#"\n\nSe presento un error al leer el archivo de configuracion "configuracion.toml". Motivo {}"#, why);
		return;
	    }
	}
	
    } else {

	config = std::sync::Arc::new(ArchivoConfiguracion::new());

	println!(r#"Archivo de configuracion "configuracion.toml" no fue encontrado ... Intentando procesar parametros de linea de comando"#);
	
	if matches.is_present("create-example-config"){
	    match ArchivoConfiguracion::crear_config_ejemplo() {
		Ok(_x) => {
		    println!(r#"\n\nArchivo "configuracion.toml" generado exitosamente!"#);
		    return;
		},
		Err(why) => {
		    println!("\n\nError al intentar generar archivo de configuracion de ejemplo: Motivo:{}", why);
		    return;
		}
	    }
	}

	if !matches.is_present("plain-mode") && !matches.is_present("tls-mode") {
	    app2.print_help().unwrap_or(());
	    println!("\n\nPor favor especificar por lo menos un modo HTTP (tls ó plain).");
	    return;
	} else if !matches.is_present("plain-mode") && matches.is_present("tls-mode") {
	    tls_mode = true;
	} else if matches.is_present("plain-mode") && !matches.is_present("tls-mode") {
	    tls_mode = false;
	} else {
	    app2.print_help().unwrap_or(());
	    println!("\n\nModo de servidor invalido. Solo se pueden escojer los modos: --tls-mode ó --plain-mode).");
	    return;
	}

	if let Some(port_value) = matches.value_of("port"){
	    port = match port_value.parse(){
		Ok(x) => x,
		Err(_why) => {
		    app2.print_help().unwrap_or(());
		    println!("\n\nNumero de puerto invalido. Valor suministrado: \"{}\"", port_value);
		    return;
		}
	    };
	}
	
	if let Some(tls_cert_name_value) = matches.value_of("tls-cert-name"){
	    
	    tls_cert_name = tls_cert_name_value.to_string();
	}
	
	if let Some(tls_cert_account_value) = matches.value_of("tls-cert-account"){
	    tls_cert_account = tls_cert_account_value.to_string();
	}
	
	
    }
    
	println!("252 \n\n");
    print!("\nIniciando Servidor Procesos de Fondo..");
   // crate::model::notificaciones_push::NotificacionPush::iniciar_motor_notificaciones(config.clone(), db_arc.clone());
    
    println!("Iniciado!\n\n");
    
    println!("\n\nIniciando servidor de aplicacion ...");
    if tls_mode == true {

	let tls_cert_url = acme_lib::DirectoryUrl::LetsEncrypt;
	let tls_cert_persist = acme_lib::persist::FilePersist::new("./tls");
	let tls_cert_dir = match acme_lib::Directory::from_url(tls_cert_persist, tls_cert_url){
	    Ok(x) => x,
	    Err(why) => {
		println!("\nFalla al intentar abrir directorio de almacenamiento para TLS. Motivo: {}",why);
		return;
	    }
	};
	
	let tls_acc = match tls_cert_dir.account(&tls_cert_account){
	    Ok(x) => x,
	    Err(why) => {
		println!("\nFalla al intentar resolver cuenta Letsencrypt para certificado TLS. motivo: {}",why);
		return;
	    }

	};
	println!("278!\n\n");
	match tls_acc.certificate(&tls_cert_name){
	    Ok(data) => {
		if let Some(tls_cert) = data {
			println!("202!\n\n");
		    warp::serve(routes::tree(db_arc, &tls_cert_name, &tls_cert_account))
			.tls()
			.key(tls_cert.private_key())
			.cert(tls_cert.certificate())
			.run(([0, 0, 0, 0], port))
			.await;
		} else {
		    println!("\n\nEl certificado TLS {} no fue encontrado en el almacenamiento de certificados. Iniciando en modo HTTP plano",tls_cert_name);

		    warp::serve(routes::tree(db_arc, &tls_cert_name, &tls_cert_account))
			.run(([0, 0, 0, 0], port))
			.await;		
		    
		}
	    },
	    Err(why) => {
		println!("\n\nFalla en leer certificado TLS del almacenamiento. Motivo: {}",why);
		return;
	    }
	};
	
    }else{
		println!("305!\n\n");
	warp::serve(routes::tree(db_arc, &tls_cert_name, &tls_cert_account))
	    .run(([0, 0, 0, 0], port))
	    .await;		
	
    }

}


#[test]
fn json_output() {

    let s = crate::model::rol::Rol::new(
	"rol1",
	"descripcion",
    );
    
    let ser : String = serde_json::to_string(&s).unwrap();
    
    println!("Esto es el resultado de la prueba: {}",ser);
}

#[test]
fn recover_structure(){

    let db = match sled::open("db"){
	Ok(db) => db,
	Err(why) => {
	    println!("\n\nCould not open database file. Reason:{}",why);
	    return;
	}
    };
    
    crate::utils::db_recovery::recover_structure(&db);
}



#[test]
fn crear_usuario(){

    let db = match sled::open("db"){
		Ok(db) => db,
		Err(why) => {
			println!("\n\nCould not open database file. Reason:{}",why);
			return;
		}
    };
    
	let mut new_user = match Usuario::crear(
		&db,
		"prueba",
		"",
		"prueba",
		"1234567896",
		"",
		"",
		"hector.viscue@grupojad.com.co",
		"hector.viscue@grupojad.com.co",
		&vec![],
		&vec![],
	){
	Ok(x) => x,
	Err(why) => {
		println!("\n\nFalla al intentar crear usario {} ", why);
		return;
	}
	};

	let sesion = crate::model::sesion::Sesion::new(&new_user);
	new_user.set_clave_acceso("12345678",&sesion);


	let mut nuevo_rol = Rol::new("Soporte", "Administrador Absoluto del Sistema");
	nuevo_rol.agregar_objetos(&mut vec![ObjetosSeguridad::ZZ999]);

	match nuevo_rol.store(&db, &sesion){
		Ok(_) => (),
		Err(why) => {
			println!("\n\nFalla al intentar crear rol para usuario {}", why);
			return;
		}
	}

	new_user.agregar_roles(&mut vec![ nuevo_rol.id() ]);

	match new_user.store(&db, &sesion) {
		Ok(()) => {
			println!("exito");
		},
		Err(_why) => {
			println!( "falla guardar usuario : {}",_why );  
		},
	}
}


fn clear_temp(config:&ArchivoConfiguracion){

/*	let config_programacion = config.programcion_limpieza_temporal.to_string();

	std::thread::spawn( move||{ 


		let mut sched = JobScheduler::new();
		match config_programacion.parse(){

			Ok( configinterpretada  )=> {

				sched.add(Job::new( configinterpretada , || {
					std::fs::remove_dir_all("./tmp").unwrap();
					std::fs::create_dir("./tmp").unwrap();
				}));
		
				loop {
					sched.tick();
					std::thread::sleep(Duration::from_millis(500));
				}
			},
			Err( error_sistema ) =>{

				dbg!( &error_sistema );
			}

		};

		
	} );*/

}
