use serde_derive::{Deserialize, Serialize};
use crate::errors;
use crate::model::{Auditable, Storable};
use crate::model::trazabilidad_plataforma::{TrazabilidadPlataforma,ObjetoModificado};
use crate::interfaces;
use crate::model::{
    rol::Rol,
    sesion::Sesion,
    objetos_seguridad::{ObjetosSeguridad, RELACION_OBJETOS_PANTALLAS},
};

#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct CrearRol{
    pub rol:String,
    pub descripcion: String,
}

pub async fn crear_rol(parametros: CrearRol, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

    let mut nuevo_rol = Rol::new(&parametros.rol, &parametros.descripcion);
    match nuevo_rol.store(&db, &sesion){
	Ok(_) => {
		let om = ObjetoModificado::Rol{id:nuevo_rol.id()};
		let mut tp = TrazabilidadPlataforma::new(&sesion.usuario(),&sesion.usuario().nombre_completo().to_string(),"Crear Rol",om,"Rol",&parametros.rol,&parametros.rol);
		tp.store(&db,&sesion).map_err(|_why|
		{
			interfaces::error_response_map( _why )
		})?;
		interfaces::return_ok_reponse( serde_json::json!({"id": nuevo_rol.id() }) )
	},
	Err(why) => interfaces::error_response( why ),
    }
    
}

#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct ActualizarRol{
    pub id:String,
    pub nombre: String,
    pub descripcion: String,
}

pub async fn actualizar_rol(parametros: ActualizarRol, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

    match Rol::load(&db, &parametros.id){
	Ok(resultado) => {
	    if let Some(mut rol) = resultado{

		rol.set_nombre(&parametros.nombre);
		rol.set_descripcion(&parametros.descripcion);

		match rol.store(&db, &sesion){
		    Ok(_) => {

				let om = ObjetoModificado::Rol{id:parametros.id.to_string()};
				let mut tp = TrazabilidadPlataforma::new(&sesion.usuario(),&sesion.usuario().nombre_completo().to_string(),"Actualizar Rol",om,"Nombre Rol",&rol.nombre(),&parametros.nombre);
				tp.store(&db,&sesion).map_err(|_why|
					{
						interfaces::error_response_map( _why )
					})?;
				interfaces::return_ok_reponse( &rol )
			},
		    Err(why) => interfaces::error_response( why ),
		}

	    }else{
		interfaces::error_response( Box::new(errors::StorableError::NotFound(parametros.id)) )
	    }
	},
	Err(why) => interfaces::error_response( why ),
	
    }
}

#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct AsignarObjetosRol{
    pub id:String,
    pub objetos: Vec<ObjetosSeguridad>,
}

pub async fn asignar_objetos_rol(mut parametros: AsignarObjetosRol, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

    match Rol::load(&db, &parametros.id){
	Ok(resultado) => {
	    if let Some(mut rol) = resultado{
			rol.agregar_objetos(&mut parametros.objetos);
			match rol.store(&db, &sesion){
				Ok(_) => interfaces::return_ok_reponse( &rol ),
				Err(why) => interfaces::error_response( why ),
			}
		    
	    }else{
		
		interfaces::error_response( Box::new(errors::StorableError::NotFound(parametros.id)) )
		    
	    }
	},
	Err(why) => interfaces::error_response( why ),
	
    }
    
}

pub async fn remover_objetos_rol(mut parametros: AsignarObjetosRol, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

    match Rol::load(&db, &parametros.id){
	Ok(resultado) => {
	    
	    if let Some(mut rol) = resultado{
			rol.remover_objetos(&mut parametros.objetos);

			match rol.store(&db, &sesion){
				Ok(_) => interfaces::return_ok_reponse( &rol ),
				Err(why) => interfaces::error_response( why ),
			}
		    
	    }else{
		
		interfaces::error_response( Box::new(errors::StorableError::NotFound(parametros.id)) )
		    
	    }
	},
	Err(why) => interfaces::error_response( why ),
	
    }
    
}

#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct EliminarRoles{
    pub ids:Vec<String>,
}

pub async fn eliminar_roles(parametros: EliminarRoles, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

    let mut deleted : Vec<String> = vec![];
    
    for mut rol in Rol::scan(&db, |x| parametros.ids.contains(&x.id()) ){
	match rol.delete(&db){
	    Ok(_) => {
			let om = ObjetoModificado::Rol{id:parametros.ids.join(",")};
			let mut tp = TrazabilidadPlataforma::new(&sesion.usuario(),&sesion.usuario().nombre_completo().to_string(),"Eliminar Rol",om,"Eliminar Rol",&rol.nombre(),"Rol Eliminado");
			tp.store(&db,&sesion).map_err(|_why|
			{
				interfaces::error_response_map( _why )
			})?;
			deleted.push(rol.id())
		}, 
	    Err(why) => return interfaces::error_response( why ),
	}
    }

    interfaces::return_ok_reponse( serde_json::json!(&deleted) )
}

#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct CargarRol{
    id:String,
}

pub async fn cargar_rol(parametros: CargarRol, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {
    match Rol::load(&db, &parametros.id){
	Ok(resultado) => {
	    if let Some(rol) = resultado{
		
		interfaces::return_ok_reponse( rol )
		    
	    }else{
		
		interfaces::error_response( Box::new(errors::StorableError::NotFound(parametros.id)) )
		    
	    }
	},
	Err(why) => interfaces::error_response( why ),
	
    }
}

pub async fn listar_roles(sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {
    let roles  = Rol::scan(&db, |x| 1==1 );
    interfaces::return_ok_reponse( serde_json::json!(&roles) )
}

#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct ListarObjetosRol{
    pub id:String,
}

pub async fn listar_objetos_asignados_rol(parametros: ListarObjetosRol, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

    match Rol::load(&db, &parametros.id){
	Ok(resultado) => {
	    if let Some(mut rol) = resultado{

		let mut objetos : Vec<std::collections::HashMap<&str,serde_json::value::Value>> = rol
		    .objetos()
		    .iter()
		    .map(|o| {
			let aux1 = o.descripcion();
			let mut aux2 = std::collections::HashMap::new();
			aux2.insert("objeto", serde_json::json!( aux1.0 ));
			aux2.insert("descripcion", serde_json::json!( aux1.1 ));
			aux2
		    })
		    .collect();

		objetos.dedup();
		
		interfaces::return_ok_reponse( objetos )
		    
	    }else{
		
		interfaces::error_response( Box::new(errors::StorableError::NotFound(parametros.id)) )
		    
	    }
	},
	Err(why) => interfaces::error_response( why ),
	
    }
    
}

pub async fn listar_objetos_noasignados_rol(parametros: ListarObjetosRol, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {
    match Rol::load(&db, &parametros.id){
	Ok(resultado) => {
	    if let Some(mut rol) = resultado{

		let mut objetos : Vec<std::collections::HashMap<&str,serde_json::value::Value>> = RELACION_OBJETOS_PANTALLAS
		    .iter()
		    .filter(|x| rol.objetos().iter().find(|y| x.1 == **y ) == None )
		    .map(|x| x.1.clone())
		    .map(|o| {
			let aux1 = o.descripcion();
			let mut aux2 = std::collections::HashMap::new();
			aux2.insert("objeto", serde_json::json!( aux1.0 ));
			aux2.insert("descripcion", serde_json::json!( aux1.1 ));
			aux2
		    })
		    .collect();

		objetos.dedup();
		
		interfaces::return_ok_reponse( objetos )
		    
	    }else{
		
		interfaces::error_response( Box::new(errors::StorableError::NotFound(parametros.id)) )
		    
	    }
	},
	Err(why) => interfaces::error_response( why ),
	
    }
}


