use crate::interfaces;
use chrono::prelude::*;
use serde_derive::{Serialize, Deserialize};
use crate::model::{Storable, Auditable};
use crate::model::{
    usuario::Usuario,
    sesion::Sesion,
    pqrs::{ 
        Pqrs,
        EstadoPqrs, 
        TipoPqrs,
    },
};
use warp::filters::multipart::{FormData, Part};
use warp::Buf;
use crate::errors;
use futures_util::TryStreamExt;
use std::cmp::Ordering;

#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct CrearPqrs{
    cod_hacienda: String,
    nombre_hacienda: String,
    telefono: String,
    email: String,
    tipo: TipoPqrs,
    asunto: String,
    mensaje: String,
	usuario_movil: String,
}

pub async fn crear_pqr(datos_pqr:CrearPqrs, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {


	let mut listado = Pqrs::scan(&db,|x| true);
	let mut count;

	if( listado.len() > 0 ){
		listado.sort_by(|a,b| if &a.id_control() < &b.id_control() { Ordering::Greater } else { Ordering::Less  });
		count= listado[0].id_control().parse::<i64>().unwrap()+1;
	}else{
		count = 1;
	}

	let mut nuevo_pqrs = Pqrs::new(
		&count.to_string(),
		&datos_pqr.cod_hacienda,
		&datos_pqr.nombre_hacienda,
		&datos_pqr.telefono,
		&datos_pqr.email,
		&datos_pqr.asunto,
		&datos_pqr.mensaje,
		EstadoPqrs::Creado,
		crate::utils::now_microseconds(),
		&datos_pqr.usuario_movil,
	);

	nuevo_pqrs.set_tipo( datos_pqr.tipo );
	nuevo_pqrs.set_created_by( &datos_pqr.usuario_movil );
	nuevo_pqrs.set_modification_stamp( 0 );
	nuevo_pqrs.set_modified_by( &"".to_string() );

	match nuevo_pqrs.store(&db, &sesion) {
		Ok(()) => {
			interfaces::return_ok_reponse( &serde_json::json!( { "id": count } ) )
		},
		Err(_why) => interfaces::error_response( _why ),
	}

}

#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct ListarPqrs{
	pub fecha_ini: i64,
	pub fecha_fin: i64,
	pub estado: EstadoPqrs,
	pub primera_carga: String,
}

pub async fn listar_pqr(filtros: ListarPqrs, _sesion: Sesion,db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {
    
	let mut result : Vec<Pqrs> =  Vec::new();

	if filtros.primera_carga.to_string() == "PrimeraVez" {
		result = Pqrs::scan(&db, |x| x.created_by().to_string() == _sesion.usuario().usuario_id().to_string() &&  ( x.estado().to_string() ==  "En Proceso" || x.estado().to_string() ==  "Creado"  ) )
		.into_iter()
		.collect();
	}
	else if filtros.estado.to_string() != "Ninguno" {

		if filtros.fecha_ini != 0{
			result  = Pqrs::scan(&db, |x| x.created_by().to_string() == _sesion.usuario().usuario_id().to_string() &&  x.estado().to_string() ==  filtros.estado.to_string() &&  crate::utils::ientre_rango(x.creation_stamp(), filtros.fecha_ini, filtros.fecha_fin) )
			.into_iter()
			.collect();
		}else{
			result = Pqrs::scan(&db, |x| x.created_by().to_string() == _sesion.usuario().usuario_id().to_string() &&  x.estado().to_string() ==  filtros.estado.to_string()  )
			.into_iter()
			.collect();
		}
	}
	else {
		result  = Pqrs::scan(&db, |x| x.created_by().to_string() == _sesion.usuario().usuario_id().to_string() &&  crate::utils::ientre_rango(x.creation_stamp(), filtros.fecha_ini, filtros.fecha_fin) )
		.into_iter()
		.collect();
	}

	result.sort_by(|a,b| if &a.id_control() < &b.id_control() { Ordering::Greater } else { Ordering::Less  });

    interfaces::return_ok_reponse( serde_json::json!(&result) )

}

pub async fn listar_todas_pqr(filtros: ListarPqrs, _sesion: Sesion,db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {
    
	let mut result : Vec<Pqrs> =  Vec::new();

	if filtros.primera_carga.to_string() == "PrimeraVez" {
		result = Pqrs::scan(&db, |x|   x.estado().to_string() ==  "En Proceso" || x.estado().to_string() ==  "Creado"   )
		.into_iter()
		.collect();
	}
	else if filtros.estado.to_string() != "Ninguno" {

		if filtros.fecha_ini != 0{
			result  = Pqrs::scan(&db, |x|  x.estado().to_string() ==  filtros.estado.to_string() &&  crate::utils::ientre_rango(x.creation_stamp(), filtros.fecha_ini, filtros.fecha_fin) )
			.into_iter()
			.collect();
		}else{
			result = Pqrs::scan(&db, |x|   x.estado().to_string() ==  filtros.estado.to_string()  )
			.into_iter()
			.collect();
		}
	}
	else {
		result  = Pqrs::scan(&db, |x|  crate::utils::ientre_rango(x.creation_stamp(), filtros.fecha_ini, filtros.fecha_fin) )
		.into_iter()
		.collect();
	}

	result.sort_by(|a,b| if &a.id_control() < &b.id_control() { Ordering::Greater } else { Ordering::Less  });

    interfaces::return_ok_reponse( serde_json::json!(&result) )

}

#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct EliminarPqrs{
    pub id_pqrs: String,
}

pub async fn eliminar_pqr(filtros: EliminarPqrs, _sesion: Sesion,db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {
    

	db.remove( filtros.id_pqrs ).map_err(|why| interfaces::error_response_map(Box::new(why)))?;

    interfaces::return_ok_reponse( serde_json::json!(  { "ejecucion": "exito" }) )

}


#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct EditarPqrs{
	id: String,
    cod_hacienda: String,
    nombre_hacienda: String,
    telefono: String,
    email: String,
    tipo: String,
    asunto: String,
    mensaje: String,
	usuario: String,
}

pub async fn editar_pqr(datos_pqr:EditarPqrs, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

	let mut pqrs_a_procesar = Pqrs::scan(&db, |x| x.id() == datos_pqr.id );

    
    for regt in pqrs_a_procesar.iter_mut(){

		let ti;

		if datos_pqr.tipo == "Felicitacion".to_string(){
			ti = TipoPqrs::Felicitacion;
		}
		else if datos_pqr.tipo == "Solicitud".to_string(){
			ti = TipoPqrs::Solicitud;
		}
		else if datos_pqr.tipo == "Reclamo".to_string(){
			ti = TipoPqrs::Reclamo;
		}
		else if datos_pqr.tipo == "Queja".to_string(){
			ti = TipoPqrs::Queja;
		}else{
			ti = TipoPqrs::Ninguno;
		}

		regt.set_cod_hacienda( &datos_pqr.cod_hacienda );
		regt.set_nombre_hacienda( &datos_pqr.nombre_hacienda );
		regt.set_telefono( &datos_pqr.telefono );
		regt.set_email( &datos_pqr.email );
		regt.set_tipo( ti);
		regt.set_asunto( &datos_pqr.asunto );
		regt.set_mensaje( &datos_pqr.mensaje );
		regt.set_modified_by( &datos_pqr.usuario);
		regt.set_modification_stamp( crate::utils::now_microseconds() );
	    regt.store(&db, &sesion).map_err(|why| Box::new(errors::StorableError::WriteError(regt.id())));
	}

	interfaces::return_ok_reponse( serde_json::json!( true ) )

}

#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct EditarEstadoPqrs{
    pub opt_estado: String,
	pub id_pqr: String,
}

pub async fn editar_estado_pqr(filtros: EditarEstadoPqrs, _sesion: Sesion,db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {
    
	let mut pqrs_a_editar = Pqrs::scan(&db, |x| x.id() == filtros.id_pqr );
	let ti;

	if filtros.opt_estado == "Creado".to_string(){
		ti = EstadoPqrs::Creado;
	}
	else if filtros.opt_estado == "EnProceso".to_string(){
		ti = EstadoPqrs::EnProceso;
	}
	else if filtros.opt_estado == "Cerrado".to_string(){
		ti = EstadoPqrs::Cerrado;
	}
	else{
		ti = EstadoPqrs::Ninguno;
	}

	
	pqrs_a_editar[0].set_estado( ti );
	match pqrs_a_editar[0].store(&db, &_sesion) {
		Ok(()) => {
			interfaces::return_ok_reponse( &serde_json::json!( { "": "" } ) )
		},
		Err(_why) => interfaces::error_response( _why ),
	}

}