use serde_derive::{Deserialize, Serialize};
use warp::{Buf, http::header::HeaderValue};
use crate::errors;
use crate::model::log_politica::{LogPolitica};
use crate::model::{
    usuario::Usuario,
    sesion::Sesion,
    refresh_token::RefreshToken,
    Storable,
};

use crate::interfaces;
use crate::utils;

pub mod cronologico;
pub mod entrada_cana;
pub mod informe_produccion;
pub mod liquidaciones_cana;
pub mod notificaciones;
pub mod politica_datos_per;
pub mod roles;
pub mod usuarios;
pub mod tls;
pub mod admin_documentos;
pub mod formato_vinculacion_natural;
pub mod formato_vinculacion_juridica;
pub mod asistencia_tecnica;
pub mod formato_autorizacion_tratamiento;
pub mod db_manager;
pub mod pqrs_admin;
    
#[derive(Deserialize, Serialize, PartialEq)]
pub struct UserCredentials{
    pub usuario: String,
    pub clave_acceso: String,
}

#[derive(Deserialize, Serialize, PartialEq)]
pub struct RespuestaUserCredentials{
    pub token: String,
    pub aceptacion: bool,
}


pub async fn autenticar(uc: UserCredentials, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

    let max_intentos_ingreso : usize = match crate::utils::ArchivoConfiguracion::leer_configuracion(){
	Ok(configuracion) => configuracion.max_intentos_inicio_sesion,
	Err(why) => 4,
    };
    
    if let Some(mut found_user) = Usuario::scan_first(&db, |item: &Usuario | {item.usuario() == uc.usuario }) {

	if max_intentos_ingreso < found_user.reintentos_acceso() {
	    return interfaces::error_response( Box::new(errors::AuthenticationError::MaxLoginAttemptExceeded) );
	}
	if found_user.bloqueo()  {

		return interfaces::error_response( Box::new(errors::AuthenticationError::InvalidUser) );
	}
	if found_user.eliminado() != None{
		return interfaces::error_response( Box::new(errors::AuthenticationError::UsuarioEliminado{user: found_user.usuario().to_string() } ) );
	}

	match found_user.verificar_clave_acceso(&uc.clave_acceso){
	    Ok(_) => {		
		let mut sesion = Sesion::new(&found_user);
                let q = sesion.clone();
		match sesion.store(&db, &q){
		    Ok(_) => {
				let respuesta = RespuestaUserCredentials{
					token: sesion.id(),
					aceptacion: found_user.aceptacion_politica_admin_datos().is_some()
				};
				interfaces::return_ok_reponse( &respuesta )
			},
		    Err(why) => interfaces::error_response( why ),
		}
	    },
	    Err(why) => {

		let maximo_intentos_nuevo = found_user.reintentos_acceso() + 1;

		found_user.set_reintentos_acceso(maximo_intentos_nuevo);
		
		let mut sesion = Sesion::new(&found_user);
		match found_user.store(&db, &sesion){
		    Ok(_)=> interfaces::error_response( Box::new(why)  ),
		    Err(why_store) => interfaces::error_response( why_store  ),
			
		}
		
		
	    },
	}
	
    } else {

	interfaces::error_response( Box::new(errors::AuthenticationError::InvalidCredentials)  )
	
    }
	
    
}

#[derive(Deserialize)]
pub struct ActualizarToken{
    pub token_firebase: String,
}


pub async fn actualizar_token(parametros: ActualizarToken, sesion: Sesion, db: std::sync::Arc<sled::Db>, config: crate::utils::ArchivoConfiguracion) -> Result<impl warp::Reply, warp::Rejection> {
    let mut usuario = Usuario::load(&db, &sesion.usuario().id())
	.map_err(|e| interfaces::error_response_map(Box::new(errors::StorableError::ReadError(e.to_string()))))?
	.ok_or( interfaces::error_response_map(Box::new(errors::StorableError::NotFound(sesion.usuario().id() ) ) ))?;

    usuario.set_token_firebase(&parametros.token_firebase);

    usuario.store(&db, &sesion)
	.map_err(|e| interfaces::error_response_map(Box::new(errors::StorableError::WriteError(e.to_string() ))))?;

    
    interfaces::return_ok_reponse(serde_json::json!({}))

}


#[derive(Deserialize, Serialize)]
pub struct SolicitudRecuperarClave{
    pub email: String,
}

#[derive(Deserialize, Serialize)]
pub struct RespuestaSolicitudRecuperarClave{
    pub token: String,
}

pub async fn solicitar_recuperar_clave(solicitud_recuperar_clave: SolicitudRecuperarClave, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {
    dbg!("paso por aqui");
	if let Some(usuario) = Usuario::scan_first(&db, |x| x.email() == solicitud_recuperar_clave.email) {

		if usuario.bloqueo()  {

			return interfaces::error_response( Box::new(errors::AuthenticationError::InvalidUser) );
		}

	}
	dbg!("paso por aqui");
    if let Some(usuario) = Usuario::scan_first(&db, |x| x.email() == solicitud_recuperar_clave.email) {
	
	let clave_dinamica = utils::generar_clave_dinamica();
	let sesion = Sesion::new(&usuario);
	let mut token = RefreshToken::new(&usuario, &clave_dinamica,&sesion);
	dbg!("paso por aqui");
	match token.store(&db, &sesion){
	    Ok(_)=>{

		let asunto = "Código Seguridad Cambio de Clave";
		let cuerpo = include_str!("solicitar_recuperar_clave.html")
		    .to_string()
		    .replace("{nombre}",&usuario.nombre_completo())
		    .replace("{usuario}",&usuario.usuario())
		    .replace("{clave}",&clave_dinamica);
		
		let respuesta = RespuestaSolicitudRecuperarClave{
		    token: token.id(),
		};
		dbg!("paso por aqui 1");
		match utils::enviar_correo(
		    &solicitud_recuperar_clave.email,
		    asunto,
		    &cuerpo
		){
		    Ok(_x) => interfaces::return_ok_reponse( &respuesta ),
		    Err(why) => interfaces::error_response( why  ),
		}
		
	    },
	    Err(why) => interfaces::error_response( why )
	}
	
    }else{
	
	interfaces::error_response( Box::new(errors::PasswordRefreshFail::InvalidEmailForRefreshingPassword(solicitud_recuperar_clave.email))  )
	    
    }
    
    
}

#[derive(Deserialize, Serialize)]
pub struct RecuperacionClave{
    pub nueva_clave: String,
    pub token: String,
    pub clave_dinamica: String,
}

pub async fn recuperar_clave(parametros: RecuperacionClave, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

    if let Some(token) = RefreshToken::scan_first(&db, |x| x.id() == parametros.token){
	
	if token.clave_dinamica() == parametros.clave_dinamica{

	    let mut usuario = token.usuario();
	    let sesion = token.sesion();
	    let _e1 = token.delete(&db);
	    
	    usuario.set_clave_acceso(&parametros.nueva_clave, &sesion);
	    usuario.set_reintentos_acceso(0);

	    match usuario.store(&db, &sesion){
		Ok(_) => {

		    let now = crate::utils::date_time_from_utc_timestamp(crate::utils::now_microseconds());
		    let fecha_actual = crate::utils::date_time_to_string(&now);
		    let asunto = "Clave de acceso modificada - Portal de Proveedores de caña Carmelita S.A.";
		    let cuerpo = include_str!("recuperar_clave.html")
			.to_string()
			.replace("{nombre}", &usuario.nombre_completo())
			.replace("{usuario}", &usuario.usuario())
			.replace("{fecha_actual}", &fecha_actual);

		    let _e2 = utils::enviar_correo(
			&usuario.email(),
			asunto,
			&cuerpo
		    );

		    let mut sesion = Sesion::new(&usuario);
		    let q = sesion.clone();
		    
		    match sesion.store(&db, &q){
			Ok(_) => {
				let respuesta = RespuestaUserCredentials{
					token: sesion.id(),
					aceptacion: usuario.aceptacion_politica_admin_datos().is_some()
				};
				interfaces::return_ok_reponse( &respuesta )
			},
			Err(why) => interfaces::error_response( why  ),
		    }
		    
		    
		},
		Err(why) => interfaces::error_response( why  ),
	    }
	    
	    
	}else{
	    interfaces::error_response( Box::new(errors::PasswordRefreshFail::InvalidDynamicPassword)  )
	}
	
    }else{
	
	interfaces::error_response( Box::new(errors::PasswordRefreshFail::PasswordRefreshTokenNotFound(parametros.token))  )
	    
    }
    
}

pub async fn check_authentication(authorization_header: HeaderValue, db: std::sync::Arc<sled::Db>  ) -> Result<(), warp::Rejection>  {

    if let Ok(session_id) = authorization_header.to_str(){
        
	match cargar_sesion_desde_id(session_id, db  ){
	    Ok(_found_session) => Ok(()),
	    Err(_why) => {
		
		Err( warp::reject::reject() )
                    
	    }
	}	
	    
    } else {
        
	Err( warp::reject::reject() )
	    
    }
    
}

pub fn cargar_sesion_desde_id(sesion_id: &str, db: std::sync::Arc<sled::Db>  ) -> Result<Sesion, Box<dyn std::error::Error>>  {
        
	match Sesion::load(&db, sesion_id){
	    Ok(resultado) => {
		
		if let Some(sesion_encontrada) = resultado{
                                                
		    Ok( sesion_encontrada )
		    
		} else {
		    
		    Err( Box::new(errors::StorableError::NotFound( sesion_id.to_string() )) )
			
		}
		    
	    },
	    Err(_why) => {
		
		Err( _why )
                    
	    }
	}
    
}



pub async fn static_upload_handler(mut body: impl Buf) -> Result<impl warp::Reply, warp::Rejection> {

    use std::io::Write;
    
    let id = utils::generate_id();

    println!("Cargando static gui...");

    let mut base : std::path::PathBuf = std::path::PathBuf::new();
    base.push("tmp");

    let mut file_name : std::path::PathBuf = base.clone();
    file_name.push( id.to_string() );
    
    let mut file = match std::fs::File::create(file_name.as_path()){
	Ok(x) => x,
	Err(_why) => return Err(warp::reject::reject()),
    };
    
    while warp::Buf::has_remaining(&body) {

	match file.write(body.bytes()) {
	    Ok(_) => (),
	    Err(_why) => return Err(warp::reject::reject()),
	};
	
	let cnt = body.bytes().len();

	println!("Cargando lote de {} bytes",cnt);
	
	warp::Buf::advance(&mut body,cnt);
    }
    
    match file.flush() {
	Ok(_) => (),
	Err(_why) => return Err(warp::reject::reject()),
    };

    println!("Carga finalizada");

    let file = match std::fs::File::open(file_name.as_path()){
	Ok(x) => x,
	Err(_why) => return Err(warp::reject::reject()),
    };

    println!("Abriendo archivo comprimido");

    let mut archive = match zip::ZipArchive::new(file){
	Ok(x) => x,
	Err(_why) => return Err(warp::reject::reject()),
    };

    for i in 0..archive.len() {
        let mut file = archive.by_index(i).unwrap();
        #[allow(deprecated)]
        let mut outpath = base.clone();
	outpath.push(file.sanitized_name());

        if (&*file.name()).ends_with('/') {
            println!(
                "File {} extracted to \"{}\"",
                i,
                outpath.as_path().display()
            );
            std::fs::create_dir_all(&outpath).unwrap();
        } else {
            println!(
                "File {} extracted to \"{}\" ({} bytes)",
                i,
                outpath.as_path().display(),
                file.size()
            );
	    
            if let Some(p) = outpath.parent() {
                if !p.exists() {
                    std::fs::create_dir_all(&p).unwrap();
                }
            }
	    
            let mut outfile = std::fs::File::create(&outpath).unwrap();
            std::io::copy(&mut file, &mut outfile).unwrap();
        }

    }

    println!("Descompresion finalizada");
	
    Ok(id)
	
}


pub async fn file_upload_handler(mut body: impl Buf) -> Result<impl warp::Reply, warp::Rejection> {

    use std::io::Write;
    
    let id = utils::generate_id();
    let file_name = format!("./tmp/{}",id);
    
    let mut file = match std::fs::File::create(&file_name){
	Ok(x) => x,
	Err(_why) => return Err(warp::reject::reject()),
    };
    
    while warp::Buf::has_remaining(&body) {

	match file.write(body.bytes()) {
	    Ok(_) => (),
	    Err(_why) => return Err(warp::reject::reject()),
	};
	
	let cnt = body.bytes().len();
	
	warp::Buf::advance(&mut body,cnt);
    }
    
    match file.flush() {
	Ok(_) => (),
	Err(_why) => return Err(warp::reject::reject()),
    };
    
    Ok(id)
	
}

pub async fn establecer_aceptacion_politica_admin_datos(sesion: Sesion, db: std::sync::Arc<sled::Db>, config: crate::utils::ArchivoConfiguracion) -> Result<impl warp::Reply, warp::Rejection> {
    let mut usuario = sesion.usuario();
    let now = crate::utils::now_microseconds();
    usuario.set_aceptacion_politica_admin_datos(now);
	let mut tp = LogPolitica::new(&usuario);
	tp.store(&db,&sesion).map_err(|_why|
	{
		interfaces::error_response_map( _why )
	})?;
    usuario.store(&db, &sesion)
	.map_err(|e| interfaces::error_response_map(Box::new(errors::StorableError::WriteError(e.to_string() ))))?;

    
    interfaces::return_ok_reponse(serde_json::json!({}))

}

pub async fn limpiar_aceptacion_politica_admin_datos(sesion: Sesion, db: std::sync::Arc<sled::Db>, config: crate::utils::ArchivoConfiguracion) -> Result<impl warp::Reply, warp::Rejection> {
    let mut usuario = sesion.usuario();
    usuario.clear_aceptacion_politica_admin_datos();

    usuario.store(&db, &sesion)
	.map_err(|e| interfaces::error_response_map(Box::new(errors::StorableError::WriteError(e.to_string() ))))?;

    
    interfaces::return_ok_reponse(serde_json::json!({}))

}


