use serde_derive::{Deserialize, Serialize};
use chrono::prelude::*;
use crate::{pdffmt,tmpfmt};
use crate::{errors, interfaces};
use crate::model::Storable;
use crate::model::{
    sesion::Sesion,
    documento::Documento,
};


#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct ListarLiquidacion{
    fecha_inicial: String,
    fecha_final: String,
    cod_hda: Option<String>,
}

pub async fn listar_liquidacion(parametros: ListarLiquidacion, sesion: Sesion, db: std::sync::Arc<sled::Db>, config: crate::utils::ArchivoConfiguracion) -> Result<impl warp::Reply, warp::Rejection> {


    #[derive(Serialize,Deserialize,Debug)]
    pub struct ConsultaLiquidacionCana{
        F_INI : String,
        F_FIN : String,
        COD_HDA: Option<String>,
        CPFS: Vec<String>,
        API_KEY:String,
    }

    #[derive(Serialize,Deserialize,Debug)]
    pub struct FilaConsultaLiquidacionCana{
	NIT: String,
        PROP: String,
        FECHA: String,
        TON_NETA: String,
        RTO: String,
        VALOR_TOTAL: String,
        VALOR_NETO: String,
        HACIENDA: String,
        TON_BRUTA: String ,
        BONIFICA_TOTAL: String,
	DOCUMENTO_ID: Option<String>,
    }

    let mut usuario = sesion.usuario();
    usuario.refresh_instance(&db);

    let peticion = ConsultaLiquidacionCana{
        COD_HDA: parametros.cod_hda,
        CPFS: usuario.nits(),
        F_INI : parametros.fecha_inicial,
        F_FIN : parametros.fecha_final,
        API_KEY: config.api_key,
    };

    
    let client = reqwest::Client::new();

    let mut res : Vec<FilaConsultaLiquidacionCana> = crate::request_proxy!(
	"{url_proxy}/liquidacion_cana",
	config.url_proxy,
	peticion,
	Vec<FilaConsultaLiquidacionCana>,
    60
    );

    for fila in res.iter_mut(){
        let fecha_procesada = DateTime::parse_from_str(&format!("{} -05:00",fila.FECHA.clone() ), "%Y-%m-%d %H:%M:%S %z")
            .map_err(|why| interfaces::error_response_map(Box::new(errors::Converciones::FalloInterpretandoFecha{origen: format!("{} 00:00:00 -05:00",fila.FECHA.clone() ), motivo: why.to_string() }) ))?;
        
        let llaves = vec![
            fila.NIT.clone(),
            fecha_procesada.format("%Y-%m-%d").to_string(),
            fila.HACIENDA.clone(),
            fila.PROP.clone()
        ];
        let doc = Documento::buscar_por_llaves(dbg!(llaves), &db).ok();
        fila.DOCUMENTO_ID = doc.map(|x| x.id() );
        fila.TON_NETA = crate::utils::redondear_str_float(&fila.TON_NETA,2);
        fila.RTO = crate::utils::redondear_str_float(&fila.RTO,2);
    }

    dbg!(&res);
        

    interfaces::return_ok_reponse(&res)

}


pub async fn listar_anticipos(parametros: ListarLiquidacion, sesion: Sesion, db: std::sync::Arc<sled::Db>, config: crate::utils::ArchivoConfiguracion) -> Result<impl warp::Reply, warp::Rejection> {

    #[derive(Serialize,Deserialize,Debug)]
    pub struct ConsultaLiquidacionAnticipos{
        CPFS:Vec<String>,
        F_INI : String,
        F_FIN : String,
        COD_HDA: Option<String>,
        API_KEY:String,
    }

    #[derive(Serialize,Deserialize,Debug)]
    pub struct FilaConsultaLiquidacionAnticipos{
	NIT: String,
        FECHA: String,
        VALOR: String,
        PREDIO: String,
        PROP: String,
	HACIENDA: String,
	DOCUMENTO_ID: Option<String>,
    }

    let mut usuario = sesion.usuario();
    usuario.refresh_instance(&db);

    let peticion = ConsultaLiquidacionAnticipos{
        COD_HDA: parametros.cod_hda,
        CPFS: usuario.nits(),
        F_INI : parametros.fecha_inicial,
        F_FIN : parametros.fecha_final,
        API_KEY: config.api_key,
    };

    let mut res : Vec<FilaConsultaLiquidacionAnticipos> = crate::request_proxy!(
	"{url_proxy}/liquidacion_anticipos",
	config.url_proxy,
	peticion,
	Vec<FilaConsultaLiquidacionAnticipos>,
	20
    );

    for fila in res.iter_mut(){
        dbg!( (&fila) );
        let fecha_procesada = DateTime::parse_from_str(&format!("{} -05:00",fila.FECHA.clone() ), "%Y-%m-%d %H:%M:%S %z")
            .map_err(|why| interfaces::error_response_map(Box::new(errors::Converciones::FalloInterpretandoFecha{origen: format!("{} 00:00:00 -05:00",fila.FECHA.clone() ), motivo: why.to_string() }) ))?;
        
        let llaves = vec![
            fila.NIT.clone(),
            fecha_procesada.format("%Y-%m-%d").to_string(),
            fila.HACIENDA.clone(),
            fila.PROP.clone()
        ];
        let doc = Documento::buscar_por_llaves(dbg!(llaves), &db).ok();
        dbg!( (&doc) );
        fila.DOCUMENTO_ID = doc.map(|x| x.id() );
    }

    dbg!(&res);

    
    interfaces::return_ok_reponse(&res)

}

pub async fn listar_ajuste_merc_exceden(parametros: ListarLiquidacion, sesion: Sesion, db: std::sync::Arc<sled::Db>, config: crate::utils::ArchivoConfiguracion) -> Result<impl warp::Reply, warp::Rejection> {

    #[derive(Serialize,Deserialize,Debug)]
    pub struct ConsultaLiquidacionMercadoExcedentario{
        F_INI : String,
        F_FIN : String,
        COD_HDA: Option<String>,
        CPFS: Vec<String>,
        API_KEY:String,
    }

    #[derive(Serialize,Deserialize,Debug)]
    pub struct FilaConsultaLiquidacionMercadoExcedentario{
	NIT: String,
        PROP: String,
        FECHA: String,
        VALOR: String,
        PREDIO: String,
	HACIENDA: String,
	DOCUMENTO_ID: Option<String>,
    }

    let mut usuario = sesion.usuario();
    usuario.refresh_instance(&db);

    let peticion = ConsultaLiquidacionMercadoExcedentario{
        COD_HDA: parametros.cod_hda,
        CPFS: usuario.nits(),
        F_INI : parametros.fecha_inicial,
        F_FIN : parametros.fecha_final,
        API_KEY: config.api_key,
    };

    
    let mut res : Vec<FilaConsultaLiquidacionMercadoExcedentario> = crate::request_proxy!(
	"{url_proxy}/liquidacion_mercado_excedentario",
	config.url_proxy,
	peticion,
	Vec<FilaConsultaLiquidacionMercadoExcedentario>,
    60
    );

    for fila in res.iter_mut(){
	
        dbg!( (&fila) );
        let llaves = vec![
            fila.NIT.clone(),
            fila.FECHA.clone(),
            fila.HACIENDA.clone(),
            fila.PROP.clone()
        ];

        let doc = Documento::buscar_por_llaves(dbg!(llaves), &db).ok();
        dbg!( (&doc) );
        fila.DOCUMENTO_ID = doc.map(|x| x.id() );
    }

    dbg!(&res);
    
    interfaces::return_ok_reponse(&res)

}
