use serde_derive::{Deserialize, Serialize};
use crate::{
    interfaces,
    errors,
    utils::{ArchivoConfiguracion, ConfiguracionBackendSICAP},
};
use crate::model::Storable;
use crate::model::{
    sesion::Sesion,
    objetos_seguridad::{ObjetosSeguridad, RELACION_OBJETOS_PANTALLAS, Pantallas},
};

#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct ListarHaciendas{
    ingreso_cana: bool,
}


pub async fn listar_haciendas(parametros:ListarHaciendas, sesion: Sesion, db: std::sync::Arc<sled::Db>, config: crate::utils::ArchivoConfiguracion) -> Result<impl warp::Reply, warp::Rejection> {

    #[derive(Deserialize, Serialize, Clone, Debug)]
    pub struct PeticionListarHaciendas{
        CPFS:Vec<String>,
        API_KEY:String,
        INGRESO_CANA: bool,
    }

    #[derive(Deserialize, Serialize, Clone, Debug)]
    pub struct FilaListarHaciendas{
        COD_HDA:String,
        NM_HDA: String,
        FCH_INI_ENT: String,
        FCH_ULT_ENT: String,
    }

    let mut usuario = sesion.usuario();
    usuario.refresh_instance(&db);

    
    let peticion = PeticionListarHaciendas{
        CPFS: usuario.nits(),
        API_KEY: config.api_key,
        INGRESO_CANA: parametros.ingreso_cana,
    };


    let res : Vec<FilaListarHaciendas> = crate::request_proxy!(
	"{url_proxy}/listar_haciendas",
	config.url_proxy,
	peticion,
	Vec<FilaListarHaciendas>,
    20
    );
    
    interfaces::return_ok_reponse(&res)
}

#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct ListarEntradaCanaGeneral{
    fecha_inicial: String,
    fecha_final: String,
    cod_hda: String,
}

pub async fn listar_entrada_cana_general(parametros: ListarEntradaCanaGeneral, sesion: Sesion, db: std::sync::Arc<sled::Db>, config: crate::utils::ArchivoConfiguracion) -> Result<impl warp::Reply, warp::Rejection> {


    let mut usuario = sesion.usuario();
    usuario.refresh_instance(&db);

    #[derive(Serialize,Deserialize,Debug)]
    pub struct FilaConsultaEntradaCanaDetalle{
        Hda: String,
        NombreHacienda: String,
        CodHda: String,
        Suerte: String,
        Hora: String,
        Guia: i64,
        VagonCanasta: String,
        PesoNeto: i64,
        VC: f64,
        Fecha: String
    }
    
    let res = crate::request_proxy_authenticated!(
        &format!("{url_proxy}/Informes/EntradaCana/{fecha_inicial}/{fecha_final}/{hacienda}/1", url_proxy=config.url_proxy, fecha_inicial= parametros.fecha_inicial ,fecha_final=parametros.fecha_final, hacienda=parametros.cod_hda),
        Vec<FilaConsultaEntradaCanaDetalle>,
        20
    );
    
    &dbg!( &res );
        
    interfaces::return_ok_reponse(&res)
}

#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct ListarEntradaCanaDetalle{
    fecha_inicial:String,
    fecha_final:String,
    cod_hda:String,
    suerte:Option<String>,
}

pub async fn listar_entrada_cana_detalle(parametros: ListarEntradaCanaDetalle, sesion: Sesion, db: std::sync::Arc<sled::Db>, config: crate::utils::ArchivoConfiguracion) -> Result<impl warp::Reply, warp::Rejection> {


    #[derive(Serialize,Deserialize,Debug)]
    pub struct ConsultaEntradaCanaDetalle{
        CPFS: Vec<String>,
        F_INI : String,
        F_FIN : String,
        COD_HDA: String,
        API_KEY:String,
        SUERTE: Option<String>,
    }

    #[derive(Serialize,Deserialize,Debug)]
    pub struct FilaConsultaEntradaCanaDetalle{
        COD_HDA: String,
        NM_HDA: String,
        LOTE: String,
        TALHAO: String,
        SUERTE: String,
        FCH_ENTRADA: String,
        HR_ENTRADA: String,
        GUIA: String,
        COD_VEHICULO: String,
        CANASTA: String,
        PESO_NETO: String,
    }

    let mut usuario = sesion.usuario();
    usuario.refresh_instance(&db);

    let peticion = ConsultaEntradaCanaDetalle{
        CPFS: usuario.nits(),
        API_KEY: config.api_key,
        F_INI : parametros.fecha_inicial,
        F_FIN : parametros.fecha_final,
        COD_HDA: parametros.cod_hda,
        SUERTE: parametros.suerte,
    };


    let mut res : Vec<FilaConsultaEntradaCanaDetalle> = crate::request_proxy!(
	"{url_proxy}/entrada_cana_detalle",
	config.url_proxy,
	peticion,
	Vec<FilaConsultaEntradaCanaDetalle>,
    20
    );

    for fila in res.iter_mut(){
	fila.PESO_NETO = crate::utils::redondear_str_float(&fila.PESO_NETO,2);
    }
    
    interfaces::return_ok_reponse(&res)
}

#[derive(Deserialize, Serialize, Clone, Debug)]
pub struct TotalGeneral{
    fecha_inicial:String,
    fecha_final:String,
    cod_hda:String,
}

pub async fn total_general(parametros: TotalGeneral, sesion: Sesion, db: std::sync::Arc<sled::Db>, config: crate::utils::ArchivoConfiguracion) -> Result<impl warp::Reply, warp::Rejection> {


    #[derive(Serialize,Deserialize,Debug)]
    pub struct TotalGeneralEntradaCana{
        CPFS: Vec<String>,
        F_INI : String,
        F_FIN : String,
        COD_HDA: String,
        API_KEY:String,
    }
    #[derive(Serialize,Deserialize,Debug)]
    pub struct RespuestaTotalGeneralEntradaCana{
        COD_HDA: String,
        TOTAL_GENERAL: String,
        TOTAL_CANASTAS: String
    }

    let mut usuario = sesion.usuario();
    usuario.refresh_instance(&db);

    let peticion = TotalGeneralEntradaCana{
        CPFS: usuario.nits(),
        API_KEY: config.api_key,
        F_INI : parametros.fecha_inicial,
        F_FIN : parametros.fecha_final,
        COD_HDA: parametros.cod_hda,
    };


    let res : RespuestaTotalGeneralEntradaCana = crate::request_proxy!(
	"{url_proxy}/total_general_entrada_cana",
	config.url_proxy,
	peticion,
	RespuestaTotalGeneralEntradaCana,
    20
    );
    
    interfaces::return_ok_reponse(&res)

}

pub async fn generar_general_pdf(db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {
    interfaces::return_ok_reponse( serde_json::json!(&false) )
}

pub async fn generar_detalle_pdf(db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {
    interfaces::return_ok_reponse( serde_json::json!(&false) )
}

