use crate::interfaces;
use chrono::prelude::*;
use serde_derive::{Serialize, Deserialize};
use crate::model::{Storable, Auditable};
use crate::model::{
    sesion::Sesion,
    usuario::Usuario,
    notificaciones_push::TipoNotificacionPush,
    notificaciones_push::NotificacionPush,
    orden_procesamiento_documentos::{
	OrdenProcesamientoDocumentos,
	EstadoDistribucion
    },
    documento::{
	TipoDocProcesado,
	Documento,
    }
};
use warp::filters::multipart::{FormData, Part};
use warp::Buf;
use crate::errors;
use futures_util::TryStreamExt;


#[derive(Deserialize)]
pub struct CrearOrden{
    pub tipo:TipoDocProcesado,
}

pub async fn crear_orden(parametros:CrearOrden, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

    let mut orden = OrdenProcesamientoDocumentos::new();
    orden.set_tipo(&parametros.tipo);

    match orden.store(&db, &sesion){
	Ok(_) => interfaces::return_ok_reponse( serde_json::json!({ "id" : orden.id()}) ),
	Err(why) => interfaces::error_response( why ),
    }
    
 
}

#[derive(Deserialize)]
pub struct EditarOrden{
    pub orden_procesamiento_id: String,
    pub tipo:TipoDocProcesado,
    pub notas: String,
}

pub async fn editar_orden(parametros:EditarOrden, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

    let mut orden = OrdenProcesamientoDocumentos::load(&db, &parametros.orden_procesamiento_id)
	.map_err(|why| interfaces::error_response_map(why))?
	.ok_or(interfaces::error_response_map(Box::new( errors::StorableError::NotFound(parametros.orden_procesamiento_id))))?;

    orden.set_tipo(&parametros.tipo);
    orden.set_notas(parametros.notas);

    orden.store(&db, &sesion)
	.map_err(|why| interfaces::error_response_map(why))?;

    interfaces::return_ok_reponse( serde_json::json!({}))
}


/*Ejemplo de un cargue funcional: 
curl 
-H 'Authorization: Sesioncc6ff52a4f388dee4ec254527cf270a0534589e6' 
-F orden_id=OrdenProcesamientoDocumentos904d93b5d1a565c31409e47030d380765cac6e03 
-F archivo=@'/home/andreshurtado/Downloads/LIQUIDACIONES_CAÑAMATA_NOVIEMBRE.pdf' 
https://proveedorescarmelita.simpleagri.com/api/admin_documentos/cargar_fuente_en_orden
*/
pub async fn cargar_fuente_en_orden(form:FormData, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

    let parts: Vec<Part> = form.try_collect().await.map_err(|e| {
        interfaces::error_response_map( Box::new(errors::ProcesamientoDocumentos::DatosMultipartInvalidos(e.to_string())) )
    })?;

    let mut file_data: Vec<u8> = vec![];
    let mut order_id_data: Vec<u8> = vec![];
	let mut content_type3: String = "".to_string();


    for mut p in parts {
        if p.name() == "archivo" {

			//let na = p.file_path();
			
            
	    /*
	    let file_ending;
            match content_type {
                Some(file_type) => match file_type {
                    "application/pdf" => {
                        file_ending = "pdf";
                    },
                    "application/zip" => {
                        file_ending = "zip";
                    },
                    v => {
                        return interfaces::error_response( Box::new( errors::ProcesamientoDocumentos::TipoArchivoIncorrecto(v.to_string()) ) )
                    },
                },
                None => {
                    eprintln!("El tipo de archivo cargado no pudo ser determinado");
                    return interfaces::error_response( Box::new( errors::ProcesamientoDocumentos::TipoArchivoIncorrecto("Tipo de archivo no se pudo determinar".to_string()) ) )
		}
	    }*/
	    
	    file_data = p
		.data()
		.await
		.ok_or( "Parte del 'contenido del archivo' no pudo ser encontrada al cargar" )
		.map_err(|e| {
		    eprintln!("Error leyendo archivo cargado: {}", e);
		    interfaces::error_response_map( Box::new( errors::ProcesamientoDocumentos::DatosMultipartNoEncontrados("archivo".to_string(), e.to_string() ) ) )
		})?
		.unwrap()
		.bytes()
		.iter()
		.map(|x| x.clone() )
		.collect();

		//content_type3 = &p.filename();//     p.data().await.map(|y| y.filename() ).collect();
		content_type3 = match p.filename() {
            Some(name_regi) => name_regi.to_string(),
            None => "No se puede encontrar un mensaje de error interno para mostrar".to_string(),
        };

	}

	if p.name() == "orden_id"{
	    
	    order_id_data = p
		.data()
		.await
		.ok_or( "Parte 'orden_id' no pudo ser encontrada" )
		.map_err(|e| {
		    eprintln!("Error leyendo archivo cargado: {}", e);
		    interfaces::error_response_map( Box::new( errors::ProcesamientoDocumentos::DatosMultipartNoEncontrados("orden_id".to_string(), e.to_string() ) ) )
		})?
		.unwrap()
		.bytes()
		.iter()
		.map(|x| x.clone())
		.collect();
	}
    }

    let order_id = std::str::from_utf8(&order_id_data).map_err(|e| interfaces::error_response_map( Box::new( errors::ProcesamientoDocumentos::ErrorConversionDatos("orden_id".to_string(), file!().to_string(), line!().to_string(), e.to_string()))) )?;

    let mut orden_procesamiento =  OrdenProcesamientoDocumentos::load(&db, order_id)
	.map_err(|why|  interfaces::error_response_map( why ) )?
	.ok_or(interfaces::error_response_map( Box::new(errors::ProcesamientoDocumentos::OrdenProcesamientoNoEncontrada(order_id.to_string())) )   )?;

    orden_procesamiento.set_fuente( &db, file_data );
    orden_procesamiento.set_estado(EstadoDistribucion::FuenteCargada( crate::utils::convert_unix_timestamp_microseconds(Utc::now().timestamp())  ));
	orden_procesamiento.set_nombre_archivo( &content_type3 );

    #[derive(Serialize)]
    struct Respuesta<'a>{
	orden_id: &'a str,
    }
		
    orden_procesamiento.store(&db, &sesion)
	.map_err(|why| interfaces::error_response_map( why ))
	.map(|y| interfaces::return_ok_reponse_map( &Respuesta{orden_id: order_id} ) )

}

#[derive(Deserialize, Clone)]
pub struct ProcesarOrden{
    pub orden_procesamiento_id:String,
}


pub async fn procesar_orden(parametros:ProcesarOrden, sesion: Sesion, db: std::sync::Arc<sled::Db>, estado_procesos_mutex: std::sync::Arc<std::sync::Mutex<std::collections::HashMap<String,String>>>) -> Result<impl warp::Reply, warp::Rejection> {

    let mut orden_procesamiento = OrdenProcesamientoDocumentos::load(&db, &parametros.orden_procesamiento_id)
	.map_err(|why| interfaces::error_response_map( why ))?
	.ok_or(interfaces::error_response_map( Box::new(errors::ProcesamientoDocumentos::OrdenProcesamientoNoEncontrada(parametros.orden_procesamiento_id.to_string())) )   )?;


    OrdenProcesamientoDocumentos::act_est_proc(orden_procesamiento.id(), &estado_procesos_mutex, "Iniciando Procesamiento...".to_string())
	.map_err(|why| interfaces::error_response_map( why ))?;
    
    match orden_procesamiento.procesar(&sesion, &db, &estado_procesos_mutex){
	Ok(_) => (),
	Err(why) =>{
	    eliminar_order_inner(parametros.clone(), sesion.clone(), db.clone())
		.map_err(|why| interfaces::error_response_map( why ))?;

	    return interfaces::error_response( why );
	}
    }

    OrdenProcesamientoDocumentos::eliminar_est_proc(orden_procesamiento.id(), &estado_procesos_mutex)
	.map_err(|why| interfaces::error_response_map( why ))?;
   
    
    interfaces::return_ok_reponse( serde_json::json!({}) )
}

#[derive(Serialize)]
pub struct ProgresoOrden{
    pub progreso:String,
}


pub async fn consulta_progreso_orden(parametros:ProcesarOrden, sesion: Sesion, db: std::sync::Arc<sled::Db>, estado_procesos_mutex: std::sync::Arc<std::sync::Mutex<std::collections::HashMap<String,String>>>) -> Result<impl warp::Reply, warp::Rejection> {


    let mensaje = OrdenProcesamientoDocumentos::consultar_est_proc(parametros.orden_procesamiento_id, &estado_procesos_mutex)
	.map_err(|why| interfaces::error_response_map(why))?;
    
    interfaces::return_ok_reponse( ProgresoOrden{
	progreso: mensaje
    } )
	
}

pub async fn consulta_progreso_orden_sse(parametros:ProcesarOrden, sesion: Sesion, db: std::sync::Arc<sled::Db>, estado_procesos_mutex: std::sync::Arc<std::sync::Mutex<std::collections::HashMap<String,String>>>) -> Result<impl warp::Reply, warp::Rejection> {


    let mensaje = OrdenProcesamientoDocumentos::consultar_est_proc(parametros.orden_procesamiento_id, &estado_procesos_mutex)
	.map_err(|why| interfaces::error_response_map(why))?;
    
    interfaces::return_ok_reponse( ProgresoOrden{
	progreso: mensaje
    } )
	
}

#[derive(Serialize, Clone)]
pub struct ResultadoConsultarDocumentosProducidos{
    documento_id: String,
    tipo: TipoDocProcesado,
    descripcion_tipo: String,
    fecha_doc: Option<i64>,
    nit:Option<String>,
    cod_hda: Option<String>,
    fecha_distribucion: Option<i64>,
    prop: Option<String>,
    distribuido: bool,
}

pub async fn consultar_documentos_producidos(parametros:ProcesarOrden, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

    let resultado =
	Documento::scan_map(&db, |(i,x)|{

	    let orden_procesamiento_id = x.orden_procesamiento_documentos_id();
	    
	    if  orden_procesamiento_id == parametros.orden_procesamiento_id{
		
		Some(
		    ResultadoConsultarDocumentosProducidos{
			documento_id: x.id(),
			tipo: x.tipo(),
			descripcion_tipo: x.tipo().to_string(),
			fecha_doc: x.fecha_documento(),
			nit:x.llaves().get(0).map(|i| i.to_owned() ),
			cod_hda: x.llaves().get(2).map(|i| i.to_owned() ),
			fecha_distribucion:x.fecha_distribucion(),
			distribuido: x.fecha_distribucion().is_some(),
			prop: x.llaves().get(3).map(|i| i.to_owned() ),
		    }
		)	    
	    }else{
		None
	    }
	});


    interfaces::return_ok_reponse( &resultado )
}

#[derive(Deserialize)]
pub struct DistribuirOrden{
    pub orden_procesamiento_id:String,
    pub preliminar: bool,
}

pub async fn distribuir_orden(parametros:DistribuirOrden, sesion: Sesion, db: std::sync::Arc<sled::Db>, config: crate::utils::ArchivoConfiguracion) -> Result<impl warp::Reply, warp::Rejection> {

    let mut orden_procesamiento = OrdenProcesamientoDocumentos::load(&db, &parametros.orden_procesamiento_id)
	.map_err(|why| interfaces::error_response_map( why ))?
	.ok_or(interfaces::error_response_map( Box::new(errors::ProcesamientoDocumentos::OrdenProcesamientoNoEncontrada(parametros.orden_procesamiento_id.to_string())) )   )?;

	orden_procesamiento.set_preliminar(parametros.preliminar);

	orden_procesamiento.store(&db,&sesion).map_err(|why| interfaces::error_response_map( why ))?;

    let mut documentos_a_procesar =
	Documento::scan(&db, |x| x.orden_procesamiento_documentos_id() == parametros.orden_procesamiento_id);

    let now : i64 = Utc::now().timestamp();
    
    for doc in documentos_a_procesar.iter_mut(){
	doc.set_fecha_distribucion(Some(now));
	doc.store(&db, &sesion)
	    .map_err(|why| interfaces::error_response_map(Box::new(errors::ProcesamientoDocumentos::FallaDistribucion(doc.id(), format!("error al almacenar fecha de distribucion en documento: {}",why)))))?;


	let nit;
	let anio;
	let llaves = doc.llaves();
	
	if let Some(niti) = llaves.get(0){
	    nit = niti;
	} else {
	    continue;
	}

	if let Some(anioi) = llaves.get(1){
	    anio = anioi;
	} else {
	    continue;
	}
	
	let tokens_destino : Vec<String> = NotificacionPush::obtener_tokens_por_nit(&db, nit);

	let title;
	let message;
	let tipo;
	
	let fecha_documento = {
	    if let Some (fecha_documento_i) = doc.fecha_documento(){
		
		let aux = crate::utils::date_time_from_utc_timestamp(fecha_documento_i);
		crate::utils::date_time_to_string(&aux)
		    
	    } else {
		
		"(No puede ser procesada la fecha del documento)".to_string()
		    
	    }
	};
	
	match doc.tipo(){
	    TipoDocProcesado::DonacionesCenicana => {
		
		title = format!("Nuevo Certificado Disponible {year}",year=&anio);
		message = format!("Estimado proveedor: Ya se encuentra disponible para descargar en la aplicación, el certificado de donación de Cenicaña correspondiente al año {year}",year=&anio );
		tipo = TipoNotificacionPush::DonacionesCenicana;
		
	    },
	    TipoDocProcesado::FondoSocial => {
		
		title = format!("Nuevo Certificado Disponible {year}",year=&anio);
		message = format!("Estimado proveedor: Ya se encuentra disponible para descargar en la aplicación, el certificado de donación del Fondo Social correspondiente al año {year}",year=&anio );
		tipo = TipoNotificacionPush::FondoSocial;
		
	    },
	    TipoDocProcesado::LiquidacionCana => {

		let preliminar : &str;
		
		if parametros.preliminar == true{
		    preliminar = "(preliminar)";
		}else{
		    preliminar = "";
		}
		title = format!("Nueva liquidación de caña disponible {fecha} {preliminar}",fecha=fecha_documento, preliminar=preliminar);
		message = format!("Estimado proveedor: Ya se encuentra disponible para descargar en la aplicación, la liquidación de caña correspondiente a {fecha}",fecha=anio );
		tipo = TipoNotificacionPush::LiquidacionCana;
		
	    },
	    TipoDocProcesado::LiquidacionMercadoExcedentario => {
		
		title = format!("Nueva liquidación de Ajuste Mercado Excedentario disponible {year}",year=&anio);
		message = format!("Estimado proveedor: Ya se encuentra disponible para descargar en la aplicación, la liquidación de Ajuste Mercado Excedentario correspondiente a {year}",year=&anio );
		tipo = TipoNotificacionPush::LiquidacionMercadoExcedentario;
		
	    },
	    TipoDocProcesado::LiquidacionAnticipos => {
		
		title = format!("Nuevo extracto de anticipos disponible {fecha}p",fecha=fecha_documento);
		message = format!("Estimado proveedor: Ya se encuentra disponible para descargar en la aplicación, el extracto de anticipos correspondiente a {fecha}",fecha=fecha_documento );
		tipo = TipoNotificacionPush::LiquidacionAnticipos;
		
	    },
	    TipoDocProcesado::Retenciones => {
		
		title = format!("Nuevo Certificado Disponible {year}",year=&anio);
		message = format!("Estimado proveedor: Ya se encuentra disponible para descargar en la aplicación, el certificado de retención de impuestos sobre la renta del año {year}", year=&anio);
		tipo = TipoNotificacionPush::Retenciones;
		
	    },
	    TipoDocProcesado::IngresosYCostos => {
		
		title = format!("Nueva Liquidación de Ingresos, Costos y Gastos Disponible {year}",year=&anio);
		message = format!("Estimado proveedor: Ya se encuentra disponible para descargar en la aplicación, la liquidación de ingresos, costos y gastos del año {year}",year=&anio);
		tipo = TipoNotificacionPush::IngresosYCostos;
		
	    },
	    TipoDocProcesado::Ica => {
		
		title = format!("Nuevo Certificado Disponible {year}",year=&anio);
		message = format!("Estimado proveedor: Ya se encuentra disponible para descargar en la aplicación, el certificado ICA del año {year}",year=&anio);
		tipo = TipoNotificacionPush::Ica;
		
	    },
		TipoDocProcesado::LiquidacionCanaParticipacion => {

			let preliminar : &str;
			
			if parametros.preliminar == true{
				preliminar = "(preliminar)";
			}else{
				preliminar = "";
			}
			title = format!("Nueva liquidación de Participación disponible {fecha} {preliminar}",fecha=fecha_documento, preliminar=preliminar);
			message = format!("Estimado proveedor: Ya se encuentra disponible para descargar en la aplicación, la liquidación de Participación correspondiente a {fecha}",fecha=anio );
			tipo = TipoNotificacionPush::LiquidacionCanaParticipacion;
			
		},
	    TipoDocProcesado::Ninguno => {
		
		title = "".to_string();
		message = "".to_string();
		tipo = TipoNotificacionPush::Ninguno;
		
	    }
	}
	
	for token in tokens_destino.iter(){

	    let mut notificacion = NotificacionPush::new(
		title.clone(),
		message.clone(),
		tipo.clone(),
		token.to_string(),
		doc.id(),
	    );

	    if let Err(why) = notificacion.enviar(&config.firebase){
		println!("Falla al enviar notificacion entrada de caña a firebase: {}", why.to_string());
	    }


	}

	
    }

    
    orden_procesamiento.set_estado(EstadoDistribucion::Distribuido( crate::utils::convert_unix_timestamp_microseconds(Utc::now().timestamp()) ));
    
    orden_procesamiento.store(&db, &sesion)
	.map_err(|e| interfaces::error_response_map(Box::new(errors::ProcesamientoDocumentos::FallaDistribucion("no se puede guardar cambio de estado iniciando proceso.".to_string(), e.to_string()))))?;

    
    interfaces::return_ok_reponse( serde_json::json!({}) )

    
}

fn eliminar_order_inner(parametros:ProcesarOrden, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<(), Box<dyn std::error::Error>> {

    let mut orden_procesamiento = OrdenProcesamientoDocumentos::load(&db, &parametros.orden_procesamiento_id)?
	.ok_or(Box::new(errors::ProcesamientoDocumentos::OrdenProcesamientoNoEncontrada(parametros.orden_procesamiento_id.to_string()))  )?;
    
    let mut documentos_a_procesar =
	Documento::scan(&db, |x| x.orden_procesamiento_documentos_id() == parametros.orden_procesamiento_id);

    
    for doc in documentos_a_procesar.iter_mut(){
	doc.delete(&db)
	    .map_err(|why| Box::new(errors::StorableError::WriteError(doc.id())))?;
	doc.delete_data(&db)?;
    }
    
    orden_procesamiento.delete(&db)?;

    Ok(())
    
}

pub async fn eliminar_orden(parametros:ProcesarOrden, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

    eliminar_order_inner(parametros, sesion, db)
	.map_err(|why| interfaces::error_response_map(why))?;
    
    interfaces::return_ok_reponse( serde_json::json!({}) )
    
}

#[derive(Serialize,Deserialize,Clone,Debug)]
pub struct ListarOrdenes{
    inicio: i64,
    fin: i64,
    usuario: Option<String>,
    tipo: Option<TipoDocProcesado>
}

#[derive(Serialize,Clone,Debug)]
pub struct ResultadoListarOrdenes{
    orden_procesamiento_documentos_id: String,
    tipo: TipoDocProcesado,
    descripcion_tipo: String,
    estado: String,
    notas: String,
	preliminar: bool,
    creation_stamp: i64,
    created_by: String, 
    modification_stamp: i64,
    modified_by: String,
	nombre_archivo: String,
}

pub async fn listar_ordenes(parametros:ListarOrdenes, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

    let lista_ordenes =
	OrdenProcesamientoDocumentos::scan_map(&db, |(i,orden)|{
	    let mut acc = true;

	    if crate::utils::ientre_rango(dbg!(orden.creation_stamp()), dbg!(parametros.inicio), dbg!(parametros.fin)) {
		acc = acc && true;
	    }else{
		acc = acc && false;
	    }

	    if let Some(usr) = &parametros.usuario {
			if crate::utils::icontiene(&Usuario::load(&db,&orden.created_by()).ok()??.usuario(), &usr) {
				acc = acc && true;
			}else{
				acc = acc && false;
			}
	    }

	    if let Some(tipo) = &parametros.tipo{
			if tipo == &orden.tipo(){
				acc = acc && true;
			}else{
				acc = acc && false
			}
	    }

	    if acc == true {
			let created_by = if let Ok(Some(usuario)) = Usuario::load(&db, &orden.created_by()){
				usuario.usuario()
			} else {
				"".to_string()
			};
			let modified_by = if let Ok(Some(usuario)) = Usuario::load(&db, &orden.modified_by()){
				usuario.usuario()
			} else {
				"".to_string()
			};
			Some(
				ResultadoListarOrdenes{
				orden_procesamiento_documentos_id: orden.id(),
				tipo: orden.tipo(),
				descripcion_tipo: orden.tipo().to_string(),
				estado: orden.estado().to_string(),
				notas: orden.notas(),
				preliminar:orden.preliminar(),
				creation_stamp: orden.creation_stamp(),
				created_by: created_by, 
				modification_stamp: orden.modification_stamp(),
				modified_by: modified_by,
				nombre_archivo: orden.nombre_archivo(),
				}
			)
	    }else{
			None
	    }
	});
    interfaces::return_ok_reponse( &lista_ordenes )
}


#[derive(Serialize,Deserialize,Clone,Debug)]
pub struct ListarDocumentos{
    inicio: i64,
    fin: i64,
    tipo: TipoDocProcesado,
}

#[derive(Serialize,Clone,Debug)]
pub struct ResultadoListarDocumentos{
    documento_id: String,
    fecha: Option<String>,
    tipo: TipoDocProcesado,
    descripcion_tipo: String,
    fecha_distribucion: Option<i64>,
    orden_procesamiento_id: String,
    prop: Option<String>,
    cod_hda: Option<String>,
    nit: Option<String>,
    creation_stamp: i64,
    created_by: String,
}


pub async fn listar_documentos(parametros:ListarDocumentos, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {
    dbg!("paso1");
    let mut usuario = sesion.usuario();
    usuario.refresh_instance(&db);

    let nits_usuario = usuario.nits();
    
    let lista_documentos =
	Documento::scan_map(&db, |(i,doc)|{
        dbg!(&doc.tipo());
	    if ! crate::utils::ientre_rango(dbg!(doc.fecha_documento()), Some(dbg!(parametros.inicio)), Some(dbg!(parametros.fin))) {
		return None;
	    }	    
        
	    if doc.tipo() != parametros.tipo{
		return None;
	    }
        dbg!(&doc);
	    if doc.fecha_distribucion() == None{
		return None;
	    }
	    
	    let nit_doc = doc.llaves().get(0)?.to_string();

	    if ! nits_usuario.iter().any(|x| x == &nit_doc){
		return None;
	    }
		
	    let created_by = Usuario::load(&db, &doc.created_by())
		.ok()?
		.map_or("".to_string(), |x| x.usuario());
		dbg!("devolver documento");
	    Some(
		ResultadoListarDocumentos{
		    documento_id: doc.id(),
		    fecha: doc.llaves().get(2).map(|x| x.to_owned() ),
		    prop: doc.llaves().get(3).map(|x| x.to_owned() ),
		    nit: doc.llaves().get(0).map(|x| x.to_owned() ),
		    cod_hda: doc.llaves().get(1).map(|x| x.to_owned() ),
		    tipo: doc.tipo(),
		    descripcion_tipo: doc.tipo().to_string(),
		    fecha_distribucion: doc.fecha_distribucion(),
		    orden_procesamiento_id: doc.orden_procesamiento_documentos_id(),
		    creation_stamp: doc.creation_stamp(),
		    created_by: created_by, 
		}
	    )

	});

    interfaces::return_ok_reponse( &lista_documentos )

}

#[derive(Deserialize)]
pub struct ObtenerDocumento{
    pub documento_id: String,
}

#[derive(Serialize)]
pub struct RespuestaObtenerLiquidacion{
    ruta: String,
}


pub async fn obtener_documento(parametros:ObtenerDocumento, sesion: Sesion, db: std::sync::Arc<sled::Db>) -> Result<impl warp::Reply, warp::Rejection> {

    use crate::{tmpzipfmt,pdffmt};
    
    let file_name = format!(pdffmt!(),crate::utils::generate_id());
    let file_path = format!(crate::tmpfmt!(), file_name);
    let write_path = std::path::Path::new(&file_path);

    let doc = Documento::load(&db, &parametros.documento_id)
	.map_err(|e| interfaces::error_response_map( e ) )?
	.ok_or(  interfaces::error_response_map(Box::new( errors::StorableError::NotFound(parametros.documento_id.clone() )) ))?;

    let data = doc.data(&db)
	.map_err(|e| interfaces::error_response_map( e ) )?;
    
    std::fs::write(&write_path,&data[..]);
    
    interfaces::return_ok_reponse( RespuestaObtenerLiquidacion{
	ruta : crate::tmpdownloadpdffmt!(file_name),
    })
    
}


